import os
from PyQt5 import QtWidgets, uic, QtCore, QtGui
from PyQt5.QtCore import QCoreApplication
from package.Config.ConfigManager import ConfigManager
from package.Service.RuleService import RuleService
from package.Helper.ClassLoader import ClassLoader
from package.Ui.Windows.CreateCheckWindow import CreateCheckWindow


class WorthScanningRulesWindow(QtWidgets.QWidget):
    closeSignal = QtCore.pyqtSignal()

    def __init__(self):
        super(WorthScanningRulesWindow, self).__init__()

        self.rulelistModel = None
        self.ruleService = RuleService()
        self.checkDropDownModel = QtGui.QStandardItemModel()
        self.createCheckWindow = CreateCheckWindow()
        self.createCheckWindow.checkInactiveSignal.connect(self.setCheckInactive)
        self.createCheckWindow.init()

        self.trans = QtCore.QTranslator(self)
        language = ConfigManager.getConfigValue("ui_language")
        self.trans.load("editWorthScanningRules_" + language + ".qm", "translations")
        QtWidgets.QApplication.instance().installTranslator(self.trans)
        uic.loadUi('gui/editWorthScanningRules.ui', self)

    def init(self):
        """
        Init all elements
        :return: None
        """
        self.rulelistModel = QtGui.QStandardItemModel()
        self.listRules.setModel(self.rulelistModel)
        self.listRules.clicked.connect(self.selectExistingRule)
        self.dropDownChecks.setModel(self.checkDropDownModel)
        self.tableViewChecks.setShowGrid(False)
        self.tableViewChecks.horizontalHeader().hide()
        self.tableViewChecks.verticalHeader().hide()
        self.buttonCreateRule.clicked.connect(self.createRule)
        self.buttonDeleteRule.clicked.connect(self.deleteRule)
        self.buttonAddCheck.clicked.connect(self.openCreateCheckWindow)
        self.initRuleList()
        self.getAvailableChecks()

    def initRuleList(self):
        """
        Displays all available rules
        :return: None
        """
        self.rulelistModel.clear()
        rules = self.ruleService.getAllRules()
        for rule in rules:
            item = QtGui.QStandardItem(QCoreApplication.translate("worth_scanning", rule['name']))
            item.setData(rule, QtCore.Qt.UserRole)
            self.rulelistModel.appendRow(item)

    def selectExistingRule(self, index):
        """
        Listener for a click on a rule. Displays all checks for the rule
        :param index:
        :return: None
        """
        item = self.rulelistModel.itemFromIndex(index)
        data = item.data(QtCore.Qt.UserRole)
        self.updateSelectedRule(data['id'])

    def updateSelectedRule(self, ruleId):
        """
        updates the displayed rule - only view related
        :param ruleId: int - id of the rule
        :return: None
        """
        self.enableAllChecks()
        rules = self.getChecksForRule(ruleId)
        self.fillRuleCheckTable(rules)
        self.setFirstActiveCheck()

    def fillRuleCheckTable(self, rules):
        """
        displays all checks for a rule
        :param rules: list
        :return: None
        """
        self.tableViewChecks.setRowCount(len(rules))
        self.tableViewChecks.setColumnCount(4)

        colmap = {
            'field': 0,
            'operator': 1,
            'value': 2,
        }

        row = 0
        for rule in rules:
            for key in rule:
                if key not in colmap:
                    continue

                column = colmap[key]

                if key == "field":
                    value = QCoreApplication.translate("Form", "check_" + rule[key])

                    # disable check in dropdown
                    self.changeCheckStatus(rule[key], False)
                elif key == "operator":
                    value = QCoreApplication.translate("Form", "operator_" + rule[key].lower())
                else:
                    valTrans = QCoreApplication.translate("Form", "option_" + rule[key].lower())
                    if rule["field"] == "biosignals" or rule["field"] == "geosignals":
                        value = rule["value"]
                    elif valTrans == 'option_' + rule[key].lower():
                        value = rule[key]
                    else:
                        value = valTrans

                self.tableViewChecks.setItem(row, column, QtWidgets.QTableWidgetItem(str(value)))

            button = QtWidgets.QPushButton(QCoreApplication.translate("Form", "button_delete"))
            button.clicked.connect(lambda state, x=rule: self.deleteCheckClick(x))
            self.tableViewChecks.setCellWidget(row, 3, button)

            row += 1

        # set column sizes
        header = self.tableViewChecks.horizontalHeader()
        for x in range(3):
            if x == 2:
                header.setSectionResizeMode(x, QtWidgets.QHeaderView.Stretch)
            else:
                header.setSectionResizeMode(x, QtWidgets.QHeaderView.ResizeToContents)

    def deleteCheckClick(self, rule):
        """
        Signal handler for clicks on the "delete check" button
        Removes check from db and from list
        :param rule: int - id of the rule
        :return: None
        """
        clickedButton = self.sender()
        if clickedButton:
            row = self.tableViewChecks.indexAt(clickedButton.pos()).row()
            self.tableViewChecks.removeRow(row)

        self.ruleService.deleteRuleCheck(rule)
        self.changeCheckStatus(rule['field'], True)

    def createRule(self):
        """
        Create a new rule
        :return: None
        """
        ruleName = self.lineEditRuleName.text()

        if ruleName == "":
            return

        if self.ruleService.checkIfRuleExists(ruleName):
            self.showErrorMessage(QCoreApplication.translate("Form", 'rule_already_exists'))
            return

        self.ruleService.createRule(ruleName)
        self.lineEditRuleName.setText("")
        self.initRuleList()

    def deleteRule(self):
        """
        Delete a rule
        :return: None
        """
        currentIndexes = self.listRules.selectedIndexes()
        for index in currentIndexes:
            data = index.data(QtCore.Qt.UserRole)
            self.ruleService.deleteRule(data['id'])

        self.initRuleList()

    def showErrorMessage(self, text):
        """
        Display an error message box
        :param text: string
        :return: None
        """
        msg = QtWidgets.QMessageBox()
        msg.setIcon(QtWidgets.QMessageBox.Critical)
        msg.setText(self.tr("Error"))
        msg.setInformativeText(text + "\n")
        msg.setWindowTitle(self.tr("Error"))
        msg.exec_()

    def getAvailableChecks(self):
        """
        Adds all available checks to the dropdown
        :return: None
        """
        checkerDir = "package/WorthScanning/Checker/"
        for moduleName in os.listdir(checkerDir):
            if moduleName.startswith("Check") and moduleName.endswith(".py"):
                packageName = "package.WorthScanning.Checker." + moduleName[0:-3]
                checkClass = ClassLoader.getClass(packageName, moduleName[0:-3])

                translationKey = "check_" + checkClass.getInternalFieldId()

                item = QtGui.QStandardItem(QCoreApplication.translate("Form", translationKey))
                item.setData(checkClass.getInternalFieldId(), QtCore.Qt.UserRole)
                self.checkDropDownModel.appendRow(item)

    def getChecksForRule(self, ruleId):
        """
        Returns a list of all checks for a rule
        :param ruleId: int, id of a rule
        :return: list
        """
        return self.ruleService.getChecksForRule(ruleId)

    def openCreateCheckWindow(self):
        """
        Opens the createCheckWindow
        :return: Nonw
        """
        ruleId = self.getSelectedRule()
        if ruleId is None:
            return

        checkType = self.dropDownChecks.currentData()
        self.createCheckWindow.setCheckType(checkType)
        self.createCheckWindow.setRuleId(ruleId)
        self.createCheckWindow.exec()

        self.updateSelectedRule(self.getSelectedRule())
        self.setFirstActiveCheck()

    def getSelectedRule(self):
        """
        returns the rule id of the selected rule in the dropdown
        :return: int
        """
        ruleId = None

        indexes = self.listRules.selectedIndexes()
        for index in indexes:
            data = index.data(QtCore.Qt.UserRole)
            ruleId = data['id']

        return ruleId

    def closeEvent(self, event):
        """
        Emits the close signal
        :param event:
        :return: None
        """
        self.closeSignal.emit()

    def enableAllChecks(self):
        """
        Sets all checks in the dropdown to enabled
        :return: None
        """
        for index in range(self.checkDropDownModel.rowCount()):
            self.checkDropDownModel.item(index).setEnabled(True)

    def setFirstActiveCheck(self):
        """
        sets the focus of the check dropdown to the first active on
        :return: None
        """
        for index in range(self.checkDropDownModel.rowCount()):
            if self.checkDropDownModel.item(index).isEnabled():
                self.dropDownChecks.setCurrentIndex(index)
                break

    def changeCheckStatus(self, field, status):
        """
        loops through all checks and changes the (in)active status of a specific check type
        :param field: string, name of check type
        :param status: bool. active or inactive
        :return: None
        """
        for index in range(self.checkDropDownModel.rowCount()):
            tmpItem = self.checkDropDownModel.item(index)
            dropdownData = tmpItem.data(QtCore.Qt.UserRole)
            if dropdownData == field:
                self.checkDropDownModel.item(index).setEnabled(status)

    def setCheckInactive(self, checkType):
        """
        signal handler for the createCheckWindow. sets a check type to inactive in the dropdown if it was added
        to a rule
        :param checkType: string - name des checks
        :return: None
        """
        self.changeCheckStatus(checkType, False)
