from PyQt5 import QtWidgets, uic, QtCore, QtGui
from package.Config.ConfigManager import ConfigManager
from package.Database.DbFactory import DbFactory
from package.Statistics.StatisticsReader import StatisticsReader
from package.Ui.Models.TableStatistics import TableStatistics


class StatisticWindow(QtWidgets.QWidget):
    closeSignal = QtCore.pyqtSignal()
    openSystemWindowSignal = QtCore.pyqtSignal(object)

    def __init__(self):
        super(StatisticWindow, self).__init__()
        self.statisticsReader = StatisticsReader(DbFactory.getDatabase())
        self.statisticsModel = None

        self.transMain = QtCore.QTranslator(self)
        language = ConfigManager.getConfigValue("ui_language")
        self.transMain.load("statisticWindow_" + language + ".qm", "translations")
        QtWidgets.QApplication.instance().installTranslator(self.transMain)

        self.transMisc = QtCore.QTranslator(self)
        self.transMisc.load("misc_" + language + ".qm", "translations")
        QtWidgets.QApplication.instance().installTranslator(self.transMisc)

        uic.loadUi('gui/statisticWindow.ui', self)

    def closeEvent(self, a0: QtGui.QCloseEvent) -> None:
        """
        Closes the Window and emits a signal. if checkers were changed open a message box
        :return: None
        """
        self.closeSignal.emit()
        self.close()

    def initStatistics(self):
        """
        Init handlers and prepare UI
        :return: None
        """
        # Statistics Buttons
        self.buttonStatisticsStars.clicked.connect(self.statisticsStarsPressed)
        self.buttonStatisticsPlanets.clicked.connect(self.statisticsPlanetsPressed)

        # StatisticsCount
        self.ddStatisticsStarCount.addItems(["5", "10", "20", "50", "100", "500", "1000", "5000"])
        self.ddStatisticsPlanetCount.addItems(["5", "10", "20", "100", "500", "1000", "5000"])

        # StatisticsSort
        for crit in self.statisticsReader.getSortOptions():
            self.ddStatisticsStarSort.addItem(self.tr(crit['text']), crit['key'])
            self.ddStatisticsPlanetSort.addItem(self.tr(crit['text']), crit['key'])

        # Criteria
        for crit in self.statisticsReader.getPlanetCriteria():
            text = QtCore.QCoreApplication.translate("additional", crit['text'])
            self.ddStatisticsPlanetCriteria.addItem(text, crit['key'])

        for crit in self.statisticsReader.getStarCriteria():
            text = QtCore.QCoreApplication.translate("additional", crit['text'])
            self.ddStatisticsStarCriteria.addItem(text, crit['key'])

        # Typen
        planetTypes = self.statisticsReader.getPlanetTypes()
        for key in planetTypes:
            self.ddStatisticsPlanetType.addItem(QtCore.QCoreApplication.translate("BodyTypes", planetTypes[key]), planetTypes[key])

        starTypes = self.statisticsReader.getStarTypes()
        for key in starTypes:
            self.ddStatisticsStarType.addItem(QtCore.QCoreApplication.translate("BodyTypes", starTypes[key]), starTypes[key])

        self.tableStatisticsView.doubleClicked.connect(self.tableDoubleClicked)

    def statisticsStarsPressed(self):
        """
        Handler for the star search button.
        :return: None
        """
        type = self.ddStatisticsStarType.currentData()
        criteria = self.ddStatisticsStarCriteria.currentData()
        sort = self.ddStatisticsStarSort.currentData()
        count = self.ddStatisticsStarCount.currentText()
        own = self.checkBoxStarOwnDiscovery.isChecked()
        onlyRings = self.checkBoxStarRings.isChecked()

        results = self.statisticsReader.getStarStatistics(type, criteria, sort, count, own, onlyRings)
        self.fillStatisticsTable(results)

    def statisticsPlanetsPressed(self):
        """
        Handler for the planet search button
        :return: None
        """
        type = self.ddStatisticsPlanetType.currentData()
        criteria = self.ddStatisticsPlanetCriteria.currentData()
        sort = self.ddStatisticsPlanetSort.currentData()
        count = self.ddStatisticsPlanetCount.currentText()
        own = self.checkBoxPlanetsOwnDiscovery.isChecked()
        landable = self.checkBoxPlanetsLandable.isChecked()
        onlyRings = self.checkBoxPlanetsRing.isChecked()

        results = self.statisticsReader.getPlanetStatistics(type, criteria, sort, count, own, landable, onlyRings)
        self.fillStatisticsTable(results)

    def fillStatisticsTable(self, data):
        """
        Add the statistic data to the table model
        :param data: list
        :return: None
        """
        # set model
        self.statisticsModel = TableStatistics(data)

        self.tableStatisticsView.setModel(None)
        self.tableStatisticsView.setModel(self.statisticsModel)

        # set column sizes
        header = self.tableStatisticsView.horizontalHeader()
        for x in range(len(header)):
            if x == 0:
                header.setSectionResizeMode(x, QtWidgets.QHeaderView.Stretch)
            else:
                header.setSectionResizeMode(x, QtWidgets.QHeaderView.ResizeToContents)

    def tableDoubleClicked(self, index) -> None:
        """
        Emit the signal if a table row is double clicked
        :param index:
        :return: None
        """
        data = self.tableStatisticsView.model().getDataForIndex(index)
        self.openSystemWindowSignal.emit(data)
