from enum import Enum
import os
from PyQt5 import QtWidgets, uic, QtCore, QtGui
from PyQt5.QtCore import QCoreApplication
from package.Config.ConfigManager import ConfigManager
from package.Config.ConfigReader import ConfigReader
from package.Database.DbFactory import DbFactory
from package.Installer.DbChecker import DbChecker
from package.Installer.DbInstaller import DbInstaller
from package.Datareader.SystemReader import SystemReader
from package.Ui.Actions import SystemAction


class SettingsWindow(QtWidgets.QWidget):
    closeSignal = QtCore.pyqtSignal()
    eventSignal = QtCore.pyqtSignal(Enum)

    def __init__(self):
        super(SettingsWindow, self).__init__()
        self.systemReader = SystemReader()

        self.trans = QtCore.QTranslator(self)
        language = ConfigManager.getConfigValue("ui_language")
        self.trans.load("settingsWindow_" + language + ".qm", "translations")
        QtWidgets.QApplication.instance().installTranslator(self.trans)
        uic.loadUi('gui/settingsWindow.ui', self)

        self.oldJournalDir = ConfigManager.getConfigValue('journal_directory')

    def initSettings(self):
        """
        Initialise all listeners
        :return: None
        """
        # Journal field
        self.settingsJournalDir.textEdited.connect(self.updateJournalDir)
        self.buttonSettingsBrowseJournal.clicked.connect(self.updateJournalDirFromFileBox)

        # Screenshots
        self.buttonScreenshotSource.clicked.connect(self.updateScreenshotSourceFromFileBox)
        self.buttonScreenshotDestination.clicked.connect(self.updateScreenshotDestinationFromFileBox)
        self.settingsConvertScreenshots.stateChanged.connect(self.updateConvertScreenshots)

        self.settingsFormatPng.format = "png"
        self.settingsFormatJpg.format = "jpg"
        self.settingsFormatJpg.toggled.connect(self.clickedScreenshotFormat)
        self.settingsFormatPng.toggled.connect(self.clickedScreenshotFormat)

        # Notifications
        self.settingsEnableDesktopNotifications.stateChanged.connect(self.updateEnableDesktopNotifications)

        # DB
        self.buttonSettingsDbSave.clicked.connect(self.changeDatabaseType)

        # language
        self.ddSettingsLanguage.addItem(QCoreApplication.translate("additional", "English"), "en")
        self.ddSettingsLanguage.addItem(QCoreApplication.translate("additional", "German"), "de")

        if ConfigManager.getConfigValue("ui_language") == "de":
            self.ddSettingsLanguage.setCurrentIndex(1)

        self.ddSettingsLanguage.currentIndexChanged.connect(self.updateLanguage)

        # home system
        self.settingsHomeName.editingFinished.connect(self.updateHomeSystemName)
        self.settingsHomePosX.editingFinished.connect(self.updateHomeSystemPosX)
        self.settingsHomePosY.editingFinished.connect(self.updateHomeSystemPosY)
        self.settingsHomePosZ.editingFinished.connect(self.updateHomeSystemPosZ)

        self.homeSuggestModel = QtWidgets.QCompleter()
        self.homeSuggestModel.setCaseSensitivity(QtCore.Qt.CaseInsensitive)
        self.settingsHomeName.setCompleter(self.homeSuggestModel)

        self.homeSuggestListModel = QtGui.QStandardItemModel()
        self.homeSuggestModel.setModel(self.homeSuggestListModel)
        self.homeSuggestModel.activated[QtCore.QModelIndex].connect(self.setHomeSystem)

        self.settingsHomeName.textEdited.connect(self.autoSuggestHomeSystem)

        # Reset credit count on sell event
        self.settingsResetCreditCountOnSellEvent.stateChanged.connect(self.updateAutomaticCreditReset)

        # temperature format
        self.ddSettingsTemperatureFormat.addItem(QCoreApplication.translate("additional", "kelvin"), "K")
        self.ddSettingsTemperatureFormat.addItem(QCoreApplication.translate("additional", "celsius"), "C")

        if ConfigManager.getConfigValue("temperature_format") == "C":
            self.ddSettingsTemperatureFormat.setCurrentIndex(1)

        self.ddSettingsTemperatureFormat.currentIndexChanged.connect(self.updateTemperatureFormat)

        # show config values
        self.showConfigValues()

        self.settingsCloseButton.clicked.connect(self.closeEvent)

    def showConfigValues(self):
        """
        Display all config values in the appropriate input fields
        :return: None
        """
        configReaderValues = ConfigReader.getConfigData()

        # Screenshot section
        self.settingsJournalDir.setText(ConfigManager.getConfigValue('journal_directory'))
        self.settingsScreenshotsSource.setText(ConfigManager.getConfigValue('screenshot_source_dir'))
        self.settingsScreenshotsDestination.setText(ConfigManager.getConfigValue('screenshot_destination_dir'))

        if ConfigManager.getConfigValue('screenshot_conversion') == "1":
            self.settingsConvertScreenshots.setChecked(True)

        if ConfigManager.getConfigValue('desktop_notifications') == "1":
            self.settingsEnableDesktopNotifications.setChecked(True)

        # screenshot format
        if ConfigManager.getConfigValue('screenshot_format') == "png":
            self.settingsFormatPng.setChecked(True)
        elif ConfigManager.getConfigValue('screenshot_format') == "jpg":
            self.settingsFormatJpg.setChecked(True)

        # Database section
        if configReaderValues["Database"]["DatabaseType"] == DbFactory.DB_MYSQL:
            self.settingsDBMysql.setChecked(True)
        elif configReaderValues["Database"]["DatabaseType"] == DbFactory.DB_SQLITE:
            self.settingsDBSqlite.setChecked(True)

        self.settingsDbServer.setText(configReaderValues["Mysql"]["Hostname"])
        self.settingsDbDatabase.setText(configReaderValues["Mysql"]["DbName"])
        self.settingsDbUser.setText(configReaderValues["Mysql"]["Username"])
        self.settingsDbPassword.setText(configReaderValues["Mysql"]["Password"])
        self.settingsDbPort.setText(configReaderValues["Mysql"]["Port"])

        self.settingsHomeName.setText(ConfigManager.getConfigValue('home_system_name'))
        self.settingsHomePosX.setText(ConfigManager.getConfigValue('home_system_pos_x'))
        self.settingsHomePosY.setText(ConfigManager.getConfigValue('home_system_pos_y'))
        self.settingsHomePosZ.setText(ConfigManager.getConfigValue('home_system_pos_z'))

        if ConfigManager.getConfigValue('automatic_credit_reset') == "1":
            self.settingsResetCreditCountOnSellEvent.setChecked(True)

    def closeEvent(self, a0: QtGui.QCloseEvent) -> None:
        """
        Closes the Window and emits a signal containing the actions that should be performed by the main window
        :return: None
        """
        self.closeSignal.emit()
        self.close()

    def updateJournalDirFromFileBox(self):
        """
        Lets the user chose a directory from a file box and saved the value
        :return:
        """
        directory = self.chooseDirectory()
        if directory != "":
            self.saveJournalDir(directory)

    def chooseDirectory(self):
        """
        Opens a file dialog and returns the selected directory
        :return:
        """
        caption = QCoreApplication.translate("SettingsForm", "Browse")
        directory = QtWidgets.QFileDialog.getExistingDirectory(self, caption, '', options=QtWidgets.QFileDialog.DontUseNativeDialog)
        return directory

    def updateLanguage(self):
        """
        Updates the chosen language
        :return:
        """
        value = self.ddSettingsLanguage.currentData()
        ConfigManager.setConfigValue("ui_language", value)

        self.eventSignal.emit(SystemAction.RELOAD_GUI)

    def updateScreenshotSourceFromFileBox(self):
        """
        Updates the source directory for the screenshot conversion
        :return: None
        """
        directory = self.chooseDirectory()
        if directory != "":
            self.settingsScreenshotsSource.setText(directory)
            ConfigManager.setConfigValue("screenshot_source_dir", directory)

    def updateScreenshotDestinationFromFileBox(self):
        """
        Updates the destination directory for the screenshot conversion
        :return: None
        """
        directory = self.chooseDirectory()
        if directory != "":
            self.settingsScreenshotsDestination.setText(directory)
            ConfigManager.setConfigValue("screenshot_destination_dir", directory)

    def clickedScreenshotFormat(self):
        """
        Updates the Screenshot format
        :return: None
        """
        button = self.sender()
        if button.isChecked():
            ConfigManager.setConfigValue("screenshot_format", button.format)

    def updateConvertScreenshots(self):
        """
        (De)Activates the screenshot conversion
        :return: None
        """
        value = 0
        if self.settingsConvertScreenshots.isChecked() == True:
            value = 1

        ConfigManager.setConfigValue("screenshot_conversion", value)

    def updateJournalDir(self):
        """
        checks the manually modified pathname for the journal directory and saves it, if it's a valid path
        :return: None
        """
        directory = self.settingsJournalDir.text()

        if os.path.exists(directory):
            self.saveJournalDir(directory)
            self.settingsJournalDir.setText(directory)

    def saveJournalDir(self, directory):
        """
        saves the setting for the journal directory

        :param directory: path as a string
        :return: None
        """
        directory = self.sanitizeDirectoryName(directory)

        ConfigManager.setConfigValue("journal_directory", directory)
        self.settingsJournalDir.setText(directory)

        self.eventSignal.emit(SystemAction.RESTART_DISCOVERY)

    def sanitizeDirectoryName(self, directory):
        """
        Checks if a path ends with a slash - adds a slash if it's missing
        :param directory: path as a string
        :return:
        """
        if directory.endswith("/") is False:
            directory += "/"

        return directory

    def updateEnableDesktopNotifications(self):
        """
        Updates the setting for the desktop configuration
        :return: None
        """
        value = 0
        if self.settingsEnableDesktopNotifications.isChecked() == True:
            value = 1

        ConfigManager.setConfigValue("desktop_notifications", value)

    def changeDatabaseType(self):
        """
        Calls the correct method to initialiste the chosen db mode
        :return: None
        """
        if self.settingsDBSqlite.isChecked():
            self.switchToSqlite()
        elif self.settingsDBMysql.isChecked():
            self.switchToMysql()

    def switchToSqlite(self):
        """
        Switches to sqlite mode. Copies all existing db settings
        :return: None
        """
        # get current config settings
        currentConfig = ConfigManager.getConfigValues()

        # save config
        ConfigReader.setValue("Database", "DatabaseType", DbFactory.DB_SQLITE)

        # reload config
        ConfigReader.loadConfig()

        # update config
        db = DbFactory.getDatabase()
        for settingType in currentConfig:
            data = {'config_value': settingType["config_value"]}
            where = {'config_name': settingType["config_name"]}
            db.update("config", data, where)

    def switchToMysql(self):
        """
        Switches to mysql mode. Copies all existing db settings
        Installs tables if they do not exist
        :return: None
        """
        dbServer = self.settingsDbServer.text()
        dbDatabase = self.settingsDbDatabase.text()
        dbUser = self.settingsDbUser.text()
        dbPassword = self.settingsDbPassword.text()
        port = self.settingsDbPort.text()

        # get current config settings
        currentConfig = ConfigManager.getConfigValues()

        # check values aka try connection
        dbChecker = DbChecker()
        if not dbChecker.checkConnection(dbServer, dbDatabase, dbUser, dbPassword, port):
            error_dialog = QtWidgets.QErrorMessage(self)
            error_dialog.setWindowModality(QtCore.Qt.WindowModal)
            error_dialog.showMessage(QCoreApplication.translate("additional", "no_database_connection"))
            error_dialog.exec_()
            return None

        # save new config
        ConfigReader.setValue("Database", "DatabaseType", DbFactory.DB_MYSQL)

        ConfigReader.setValue("Mysql", "Hostname", dbServer)
        ConfigReader.setValue("Mysql", "DbName", dbDatabase)
        ConfigReader.setValue("Mysql", "Username", dbUser)
        ConfigReader.setValue("Mysql", "Password", dbPassword)
        ConfigReader.setValue("Mysql", "Port", port)

        # reload config
        ConfigReader.loadConfig()

        # reset database connection
        DbFactory.resetConnection()

        # create mysql tables if necessary
        if not dbChecker.checkIfTablesExist():
            installer = DbInstaller()
            installer.install()

        # update config
        db = DbFactory.getDatabase()
        for settingType in currentConfig:
            data = {'config_value': settingType["config_value"]}
            where = {'config_name': settingType["config_name"]}
            db.update("config", data, where)

    def updateHomeSystemName(self):
        """
        Updates the home system name
        :return: None
        """
        value = self.settingsHomeName.text()
        ConfigManager.setConfigValue("home_system_name", value)

    def updateHomeSystemPosX(self):
        """
        Updates the home system position x
        :return: None
        """
        value = self.settingsHomePosX.text()
        ConfigManager.setConfigValue("home_system_pos_x", value)

    def updateHomeSystemPosY(self):
        """
        Updates the home system position y
        :return: None
        """
        value = self.settingsHomePosY.text()
        ConfigManager.setConfigValue("home_system_pos_y", value)

    def updateHomeSystemPosZ(self):
        """
        Updates the home system position z
        :return: None
        """
        value = self.settingsHomePosZ.text()
        ConfigManager.setConfigValue("home_system_pos_z", value)

    def getSystemNames(self, name):
        """
        Gets system namens from the db for the home system auto suggest
        :param name: string, content of the system name input field
        :return: None
        """
        if len(name) < 3:
            return {}

        return self.systemReader.searchSystemByName(name)

    def autoSuggestHomeSystem(self):
        """
        Selects system names and adds them to the autosuggest
        :return: None
        """
        text = self.settingsHomeName.text()
        result = self.getSystemNames(text)

        self.homeSuggestListModel.clear()

        for entry in result:
            item = QtGui.QStandardItem(entry['system_name'])
            item.setData(entry, QtCore.Qt.UserRole)
            self.homeSuggestListModel.appendRow(item)

    def setHomeSystem(self, index):
        """
        Saves the new home system coordinates and triggers a signal to update homesystem in observer
        :param index: Index of the selected autosuggest element
        :return: None
        """
        itemData = index.data(QtCore.Qt.UserRole)

        self.settingsHomePosX.setText(str(itemData['pos_x']))
        self.settingsHomePosY.setText(str(itemData['pos_y']))
        self.settingsHomePosZ.setText(str(itemData['pos_z']))

        self.updateHomeSystemName()
        self.updateHomeSystemPosX()
        self.updateHomeSystemPosY()
        self.updateHomeSystemPosZ()

        self.eventSignal.emit(SystemAction.UPDATE_HOME_SYSTEM)

    def updateAutomaticCreditReset(self) -> None:
        """
        Updates the setting for the automatic credit reset
        :return: None
        """
        value = 0
        if self.settingsResetCreditCountOnSellEvent.isChecked() == True:
            value = 1

        ConfigManager.setConfigValue("automatic_credit_reset", value)

    def updateTemperatureFormat(self):
        """
        Updates the chosen temperature format
        :return:
        """
        value = self.ddSettingsTemperatureFormat.currentData()
        ConfigManager.setConfigValue("temperature_format", value)
