from PyQt5 import QtWidgets, uic, QtCore, QtGui
from PyQt5.QtCore import QSettings, QPoint
from PyQt5.QtCore import QCoreApplication
from functools import partial
from package.Config.ConfigManager import ConfigManager
from package.Service.SearchService import SearchService
from package.Ui.Element.SystemBody import SystemBody
from package.Ui.Element.SystemInfo import SystemInfo
from package.Ui.Element.PlanetInfo import PlanetInfo
from package.Ui.Element.StarInfo import StarInfo
from package.Journal.Events.BodyEntity import BodyEntity
from package.Observer.SystemState import SystemState
from package.Observer.Analyzer import Analyzer
from package.Ui.Element.SystemScanBox import SystemScanBox
from package.Observer.ObserverMessage import ObserverMessage


class SearchWindow(QtWidgets.QWidget):
    closeSignal = QtCore.pyqtSignal()

    def __init__(self):
        super(SearchWindow, self).__init__()
        self.searchSuggestModel = None
        self.searchSuggestListModel = None
        self.searchService = SearchService()
        self.maxLeftScrollSize = 0
        self.system = None
        self.systemAddress = None
        self.bodyId = None
        self.systemState = SystemState()
        self.analyzer = Analyzer.getInstance()
        self.systemScanBox = SystemScanBox()
        self.widgetList = {}
        self.widgetCount = 0

        self.trans = QtCore.QTranslator(self)
        language = ConfigManager.getConfigValue("ui_language")
        self.trans.load("searchwindow_" + language + ".qm", "translations")
        QtWidgets.QApplication.instance().installTranslator(self.trans)

        uic.loadUi('gui/searchWindow.ui', self)

        # restore window position
        self.settings = QSettings('timeagent', 'eliza-search')
        self.move(self.settings.value("pos", QPoint(50, 50)))

    def closeEvent(self, a0: QtGui.QCloseEvent) -> None:
        """
        Closes the Window and emits a signal containing the actions that should be performed by the main window
        :return: None
        """
        self.settings.setValue("pos", self.pos())
        self.closeSignal.emit()
        self.close()

    def setSystemAddress(self, systemAddress: int) -> None:
        """
        Set the system address
        :param systemAddress: int
        :return: None
        """
        self.systemAddress = systemAddress

    def setBodyId(self, bodyId: int) -> None:
        """
        Set the body id
        :param bodyId: int
        :return: None
        """
        self.bodyId = bodyId

    def init(self) -> None:
        """
        init all elements
        :return:
        """
        # auto suggest search
        self.searchSuggestModel = QtWidgets.QCompleter()
        self.searchSuggestModel.setCaseSensitivity(QtCore.Qt.CaseInsensitive)
        self.lineEditSearch.setCompleter(self.searchSuggestModel)

        self.searchSuggestListModel = QtGui.QStandardItemModel()
        self.searchSuggestModel.setModel(self.searchSuggestListModel)
        self.searchSuggestModel.activated[QtCore.QModelIndex].connect(self.suggestActivated)

        self.lineEditSearch.textEdited.connect(self.triggerAutoSuggestSearch)
        self.lineEditSearch.returnPressed.connect(self.triggerTextSearch)

        self.rightContent.addStretch(1)

        self.systemScanLayout.addWidget(self.systemScanBox)
        self.systemScanBox.hide()

        self.initSystem()

    def initSystem(self) -> None:
        """
        if a system address is set, init the system and body view
        :return: None
        """
        self.clearAllLayouts()
        # prefilled values?
        if self.systemAddress is not None:
            self.displaySystem(self.systemAddress)

            if self.bodyId is not None:
                body = self.system.getBodyById(self.bodyId)
                self.displayBodyInfo(body)
                self.adjustBackgroundColors(self.bodyId)
                self.setScrollbarPosition()

    def triggerAutoSuggestSearch(self) -> None:
        """
        does the auto suggest search
        :return: None
        """
        searchTerm = self.lineEditSearch.text()

        if len(searchTerm) < 3:
            return

        result = self.searchService.getAutoSuggestResult(searchTerm)

        self.searchSuggestListModel.clear()

        for entry in result:
            name = entry['system_name']
            if 'planet_name' in entry:
                name = entry['planet_name']

            item = QtGui.QStandardItem(name)
            item.setData(entry, QtCore.Qt.UserRole)
            self.searchSuggestListModel.appendRow(item)

    def triggerTextSearch(self) -> None:
        """
        handler if search was executed with return button
        :return: None
        """
        searchTerm = self.lineEditSearch.text()

        if len(searchTerm) < 3:
            return

        result = self.searchService.getAutoSuggestResult(searchTerm)

        if len(result) == 0:
            return

        self.processSearch(result[0])

    def suggestActivated(self, index) -> None:
        """
        handler if auto suggest element is selected
        :param index:
        :return: None
        """
        itemData = index.data(QtCore.Qt.UserRole)
        self.processSearch(itemData)
        self.setScrollbarPosition()

    def processSearch(self, itemData: dict) -> None:
        """
        clear layouts and load system data
        :param itemData: dict
        :return: None
        """
        if 'system_address' in itemData:
            # clear layouts
            self.clearAllLayouts()
            self.systemScanBox.hide()
            self.systemState.reset()
            self.leftScrollArea.verticalScrollBar().setValue(0)

            self.displaySystem(itemData['system_address'])

            self.bodyId = None
            if 'body_id' in itemData:
                body = self.system.getBodyById(itemData['body_id'])
                self.displayBodyInfo(body)
                self.adjustBackgroundColors(itemData['body_id'])
                self.bodyId = itemData['body_id']

    def clearAllLayouts(self) -> None:
        """
        clear all layouts
        :return: None
        """
        self.clearLayout(self.leftContent)
        self.clearLayout(self.systemLayout)
        self.clearLayout(self.planetLayout)

    def displaySystem(self, systemAddress: int) -> None:
        """
        display the basic system infos (bodies on left, system info center)
        :param systemAddress: int
        :return: None
        """
        self.system = self.searchService.getSystem(systemAddress)
        self.displaySystemList()
        self.displaySystemInfo()

    def clearLayout(self, layout) -> None:
        """
        remove all content inside the given layout
        :param layout:
        :return: None
        """
        if layout is not None:
            while layout.count():
                child = layout.takeAt(0)
                if child.widget() is not None:
                    child.widget().deleteLater()
                elif child.layout() is not None:
                    self.clearLayout(child.layout())

    def displaySystemList(self) -> None:
        """
        call function to display the bodies in the system list and adjust size of scroll area
        :return: None
        """
        self.maxLeftScrollSize = 0
        self.widgetCount = 0
        self.displaySystemBodies(self.system.getSystemTree())

        # adjust left scroll area size
        self.leftScrollArea.setMinimumWidth(self.maxLeftScrollSize + 50)
        self.leftContent.addStretch(1)

        # update scrollbar positions after widgets are inserted
        QCoreApplication.processEvents()

    def displaySystemBodies(self, tree) -> None:
        """
        display system bodies
        :param tree:
        :return: None
        """
        for bodyId in tree:
            object = tree[bodyId]['object']
            elm = SystemBody(None, object)
            elm.clickSignal.connect(self.displaySystemAndBodyData)

            self.widgetCount += 1
            self.widgetList[bodyId] = self.widgetCount

            if self.maxLeftScrollSize < elm.sizeHint().width():
                self.maxLeftScrollSize = elm.sizeHint().width()

            self.leftContent.addWidget(elm)

            if "childs" in tree[bodyId] and len(tree[bodyId]['childs']) > 0:
                self.displaySystemBodies(tree[bodyId]['childs'])

            # add body to systemstate
            self.systemState.addBody(object)

    def displaySystemAndBodyData(self, body: BodyEntity) -> None:
        """
        Shortcut method to display system and body info
        :param body: Planet or Star (BodyEntity)
        :return: None
        """
        self.adjustBackgroundColors()

        self.displaySystemInfo()
        self.displayBodyInfo(body)

    def displaySystemInfo(self) -> None:
        """
        Display the SystemInfo element
        :return: None
        """
        self.clearLayout(self.systemLayout)

        elm = SystemInfo(self, self.system)
        self.systemLayout.addWidget(elm)
        self.systemLayout.addStretch(1)

        self.analizeSystem()

    def displayBodyInfo(self, body: BodyEntity) -> None:
        """
        Display the PlanetInfo or StarInfo element
        :param body: Planet or Star (BodyEntity)
        :return: None
        """
        self.clearLayout(self.planetLayout)

        if body.isPlanet():
            elm = PlanetInfo(self, body)
            self.planetLayout.addWidget(elm)
        elif body.isStar():
            elm = StarInfo(self, body)
            self.planetLayout.addWidget(elm)

        self.planetLayout.addStretch(1)

    def analizeSystem(self) -> None:
        """
        start the analyzer for the system and show/hide the system scan box
        :return:
        """
        results = self.analyzer.analyzeSystem(self.systemState)

        if len(results) > 0:
            self.systemScanBox.clearMessages()
            self.systemScanBox.show()
            for message in results:
                if message.destination == ObserverMessage.OBSERVER_DESTINATION_CHECKLOG and message.data.found == True:
                    self.systemScanBox.addMessage(message.data.body)
                    self.systemScanBox.addMessage(message.data.message)
                    self.systemScanBox.addMessage("")
        else:
            self.systemScanBox.hide()

    def adjustBackgroundColors(self, bodyId: int = None) -> None:
        """
        Remove background color from system bodys and mark the active one
        :param bodyId: int
        :return: None
        """
        items = (self.leftContent.itemAt(i).widget() for i in range(self.leftContent.count()))
        for widget in items:
            if isinstance(widget, SystemBody):
                widget.setAutoFillBackground(False)

                if bodyId is not None and widget.object.getBodyId() == bodyId:
                    widget.setAutoFillBackground(True)

    def setScrollbarPosition(self) -> None:
        """
        updates the scrollbar position after the system is displayed. scroll to top if no body is selected
        or show active body widget
        see: https://stackoverflow.com/questions/52450219/qscrollarea-ensurewidgetvisible-method-does-not-show-target-widget
        :return: None
        """
        if self.bodyId is not None:
            widgetNumber = self.widgetList[self.bodyId]
            lastWidget = self.leftContent.itemAt(widgetNumber - 1).widget()
            self.show()
            QtCore.QTimer.singleShot(0, partial(self.leftScrollArea.ensureWidgetVisible, lastWidget))
        else:
            self.leftScrollArea.verticalScrollBar().setValue(0)
