from enum import Enum
from PyQt5 import QtWidgets, uic, QtCore, QtGui
from PyQt5.QtCore import QCoreApplication
from package.Config.ConfigManager import ConfigManager
from package.Database.DbFactory import DbFactory
from package.Import.ImportHandler import ImportHandler
from package.Installer.DbReseter import DbReseter
from package.Observer.CalculationHandler import CalculationHandler
from package.Ui.Actions import SystemAction
from package.Ui.Handler.ObserverHandler import ObserverHandler as UiObserverHandler


class DbManagementWindow(QtWidgets.QWidget):
    closeSignal = QtCore.pyqtSignal()
    eventSignal = QtCore.pyqtSignal(Enum)

    def __init__(self):
        super(DbManagementWindow, self).__init__()
        self.uiObserverHandler = UiObserverHandler(self)
        self.importHandler = None
        self.dbThread = None
        self.reseter = None

        self.trans = QtCore.QTranslator(self)
        language = ConfigManager.getConfigValue("ui_language")
        self.trans.load("dbManagementWindow_" + language + ".qm", "translations")
        QtWidgets.QApplication.instance().installTranslator(self.trans)
        uic.loadUi('gui/dbManagementWindow.ui', self)

    def closeEvent(self, a0: QtGui.QCloseEvent) -> None:
        """
        Closes the Window
        :return: None
        """
        self.closeSignal.emit()
        self.close()

    def init(self):
        """
        Add listener to buttons
        :return: None
        """
        self.buttonDmCreate.clicked.connect(self.dbManagementCreatePressed)
        self.buttonDmReset.clicked.connect(self.dbManagementResetPressed)

    def dbManagementCreatePressed(self):
        """
        Starts the import
        :return: None
        """
        if self.uiObserverHandler.checkObserverSettings() is False:
            self.showErrorMessage(QCoreApplication.translate("additional", "journal_directory_missing"))
            return

        if self.isImportRunning():
            self.dbManagementStop()
            return

        # stop discovery server
        self.eventSignal.emit(SystemAction.STOP_DISCOVERY)

        # change text
        self.buttonDmCreate.setText(QCoreApplication.translate("additional", "stop_import"))

        self.progressBarDmFile.setValue(0)
        self.progressBarDmAll.setValue(0)
        self.textBrowserDm.clear()

        # create thread
        self.dbThread = QtCore.QThread()

        # create threaded object
        self.importHandler = ImportHandler()
        self.importHandler.moveToThread(self.dbThread)
        self.importHandler.updateTextSignal.connect(self.displayUpdate)
        self.importHandler.updateBarFileSignal.connect(self.updateProgressBarFile)
        self.importHandler.updateBarAllSignal.connect(self.updateProgressBarAll)
        self.importHandler.errorSignal.connect(self.dbManagementError)
        self.importHandler.finishedSignal.connect(self.dbManagementStop)
        self.dbThread.started.connect(self.importHandler.run)
        self.dbThread.start()

    def dbManagementError(self, errorMessage):
        """
        Handler for opening a error box based on a signal
        :param errorMessage:
        :return: None
        """
        self.showErrorMessage(errorMessage)
        self.dbManagementStop()

    def showErrorMessage(self, text):
        """
        Display a messagebox with a error message
        :param text:
        :return: None
        """
        msg = QtWidgets.QMessageBox()
        msg.setIcon(QtWidgets.QMessageBox.Critical)
        msg.setText(self.tr("Error"))
        msg.setInformativeText(text + "\n")
        msg.setWindowTitle(self.tr("Error"))
        msg.exec_()

    def isImportRunning(self):
        """
        Checks if the import is running
        :return: bool
        """
        if isinstance(self.dbThread, QtCore.QThread):
            return True

        return False

    def dbManagementStop(self):
        """
        Stops the import if it is running
        :return: None
        """
        if self.importHandler and self.dbThread:
            self.importHandler.stop()
            self.dbThread.quit()
            self.dbThread.wait()
            self.dbThread = None

            self.buttonDmCreate.setText(QCoreApplication.translate("DBManagement", "start_import"))
            self.eventSignal.emit(SystemAction.START_DISCOVERY)

    def displayUpdate(self, message):
        """
        appends text to the textbox
        :param message: text
        :return: None
        """
        self.textBrowserDm.append(message)

    def updateProgressBarFile(self, value):
        """
        Update the "file" progress bar
        :param value: int
        :return: None
        """
        self.progressBarDmFile.setValue(value)

    def updateProgressBarAll(self, value):
        """
        update the "all" progress bar
        :param value: int
        :return: None
        """
        self.progressBarDmAll.setValue(value)

    def dbManagementResetPressed(self):
        """
        handler for reset button. displays a messagebox and starts the reset
        :return: None
        """
        msg = QtWidgets.QMessageBox()
        result = msg.question(self, '', QCoreApplication.translate("additional", "question_reset_database"), msg.Yes | msg.No)
        if result == msg.Yes:
            self.resetDatabase()

    def resetDatabase(self):
        """
        Will reset all db data (using a thread)
        :return: None
        """
        # clear text window
        self.textBrowserDm.clear()
        self.displayUpdate(QCoreApplication.translate("additional", "reset_database_start"))
        self.progressBarDmAll.setValue(1)

        # create thread
        self.dbThread = QtCore.QThread()

        # create threaded object
        self.reseter = DbReseter()
        self.reseter.moveToThread(self.dbThread)
        self.reseter.updateBarAllSignal.connect(self.updateProgressBarAll)
        self.reseter.finishedSignal.connect(self.finishedReset)
        self.dbThread.started.connect(self.reseter.reset)
        self.dbThread.start()

    def finishedReset(self):
        """
        stops the reset db thread and resets credit counts
        :return: None
        """
        self.dbThread.quit()
        self.dbThread.wait()
        self.dbThread = None

        # reset session credits
        calc = CalculationHandler.getInstance(DbFactory.getDatabase())
        calc.resetValues()
        self.eventSignal.emit(SystemAction.DISCOVERY_RESET_DATA)

        self.displayUpdate(QCoreApplication.translate("additional", "reset_database_finished"))
