import re
from PyQt5 import QtWidgets, uic, QtCore
from PyQt5.QtCore import QCoreApplication
from package.Config.ConfigManager import ConfigManager
from package.Helper.ClassLoader import ClassLoader
from package.Database.DbFactory import DbFactory


class CheckerConfigWindow(QtWidgets.QDialog):
    def __init__(self, checkerName: str):
        super(CheckerConfigWindow, self).__init__()

        self.checkerName = checkerName
        self.checker = None
        self.values = {}

        language = ConfigManager.getConfigValue("ui_language")
        self.trans = QtCore.QTranslator(self)
        self.trans.load("editCheckerValues_" + language + ".qm", "translations")
        QtWidgets.QApplication.instance().installTranslator(self.trans)
        uic.loadUi('gui/editCheckerValues.ui', self)

        self.initCheckFields()

    def initCheckFields(self) -> None:
        """
        Init window, display all config value data
        :return: None
        """
        self.checkerSaveButton.clicked.connect(self.saveValues)

        self.checker = self.getCheckerClass()
        configFields = self.checker.getConfigFields()

        self.descriptionLabel.setText(QCoreApplication.translate("Content", self.checkerName + "_headline"))
        self.descriptionLabel.setWordWrap(True)

        for field in configFields:
            elm = self.getConfigElement(field)
            self.checkerValuesLayout.addLayout(elm)

    def getCheckerClass(self):
        """
        Returns a checker class
        :return: CheckerAbstract
        """
        packageNameBase = "package.Observer.Checker."
        packageName = packageNameBase + self.checkerName
        checkerClass = ClassLoader.getClass(packageName, self.checkerName)

        return checkerClass

    def getConfigElement(self, field: dict):
        """
        Create a config value UI element.
        It's simply a hlayout containing a label and a form field
        :param field: dict
        :return: QtWidget/Layout
        """
        mainLayout = QtWidgets.QHBoxLayout()

        # create label
        textlabel = QtWidgets.QLabel()
        textlabel.setText(QCoreApplication.translate("Content", self.checkerName + "_" + field["name"]))
        mainLayout.addWidget(textlabel)

        self.values[field["name"]] = self.getInputField(field["type"], field["value"])
        mainLayout.addWidget(self.values[field["name"]])

        return mainLayout

    def getInputField(self, type: str, default: str):
        """
        Return the UI field type for a config value
        :param type: string
        :param default: string
        :return: QtWidget
        """
        if type == "float" or type == "int":
            widget = QtWidgets.QLineEdit()
            widget.setText(default)

        return widget

    def saveValues(self) -> None:
        """
        Triggered when the save button is clicked.
        Validate all config values and insert them if they are valid
        :return: None
        """
        if not self.validateValues():
            return

        self.insertValues()
        self.close()

    def validateValues(self) -> bool:
        """
        Loop over all config values and validate them according to their type
        :return: bool
        """
        allValid = True
        for field in self.checker.getConfigFields():
            name = field['name']

            if field['type'] == "float":
                isValid = self.checkFloatValue(self.values[name].text())
            elif field['type'] == "int":
                isValid = self.checkIntValue(self.values[name].text())
            else:
                isValid = True

            if not isValid:
                allValid = False
                self.values[name].setStyleSheet("border: 1px solid red;")
            else:
                self.values[name].setStyleSheet("")

        return allValid

    def insertValues(self) -> None:
        """
        Loop over all config values and insert them in the db
        :return: None
        """
        for field in self.checker.getConfigFields():
            value = None
            name = field['name']

            if field['type'] == "float" or field['type'] == "int":
                value = self.values[name].text()
            elif field['type'] == "dropdown":
                value = self.values[name].currentData()

            self.saveValue(name, value)

    def checkFloatValue(self, value: str) -> bool:
        """
        Check if the value is a float value
        :param value: string
        :return: None
        """
        isValid = False
        regex = re.compile('^[0-9.]+$')
        if regex.search(value):
            isValid = True

        return isValid

    def checkIntValue(self, value: str) -> bool:
        """
        Check if the value is an int value
        :param value: string
        :return: bool
        """
        isValid = False
        regex = re.compile('^[0-9]+$')
        if regex.search(value):
            isValid = True

        return isValid

    def saveValue(self, configName: str, configValue: str) -> None:
        """
        Save a config value in db
        :param configName: string
        :param configValue: string
        :return: None
        """
        db = DbFactory.getDatabase()

        updateData = {'config_value': configValue}
        where = {
            'checker_name': self.checkerName,
            'config_name': configName
        }

        db.update("checker_configuration", updateData, where)
