from datetime import datetime
import pytz
import locale
from PyQt5 import QtCore, QtGui, QtWidgets
from PyQt5.QtCore import Qt
from PyQt5.QtCore import QCoreApplication
from package.Datareader.FlightlogReader import FlightlogReader
from package.Elite.StarIconMap import STAR_ICON_MAP


class TableFlightLog(QtCore.QAbstractTableModel):
    columnMapping = {
        0: "flightlog_system",
        1: "flightlog_distance",
        2: "flightlog_date",
    }
    BATCH_SIZE = 20

    def __init__(self, data = []):
        super(TableFlightLog, self).__init__()
        self.reader = FlightlogReader()
        self.allJumpsCount = self.reader.getJumpCount()
        self.currentJumpCount = 0
        self._data = []
        self.loadData()
        self.currentJumpCount = self.BATCH_SIZE
        self.datetimeFormatDest = '%d.%m.%Y %H:%M:%S'

    def rowCount(self, index=None):
        return self.currentJumpCount

    def columnCount(self, index=None):
        if len(self._data) > 0:
            return 3
        else:
            return 0

    def getColumnNameForIndex(self, index):
        return self.columnMapping[index]

    def data(self, index, role=None):
        if not index.isValid():
            return None

        if index.row() >= len(self._data) or index.row() < 0:
            return None

        mappedIndex = self.getColumnNameForIndex(index.column())
        value = self._data[index.row()][mappedIndex]
        if role == Qt.DisplayRole:
            if mappedIndex == "flightlog_date":
                if isinstance(value, datetime):
                    return value.replace(tzinfo=pytz.utc).astimezone().strftime(self.datetimeFormatDest)
                else:
                    return value
            elif mappedIndex == "flightlog_distance":
                value = locale.format_string('%.4f', value, True)
                return value + " " + QCoreApplication.translate("Flightlog", "lightyear_abbr")
            elif isinstance(value, float):
                return "%.2f" % value
            else:
                return str(self._data[index.row()][mappedIndex])

        if role == QtCore.Qt.DecorationRole and index.column() == 0:
            starType = self._data[index.row()]['flightlog_type']
            if starType is not None:
                starImageName = self.getStarIcon(starType)
                path = "gui/images/stars/" + starImageName
                pixmap = QtGui.QPixmap(path)
                return pixmap

        if role == QtCore.Qt.ToolTipRole and index.column() == 0:
            starType = self._data[index.row()]['flightlog_type']
            if starType is not None:
                return QCoreApplication.translate("BodyTypes", starType)

    def headerData(self, section, orientation, role=Qt.DisplayRole):
        if len(self._data) == 0:
            return None

        if role == Qt.DisplayRole and orientation == Qt.Horizontal:
            return QCoreApplication.translate("Flightlog", self.columnMapping[section])

    def flags(self, index):
        return QtCore.Qt.ItemIsEnabled

    def canFetchMore(self, parent):
        if parent.isValid():
            return False

        return self.currentJumpCount < self.allJumpsCount

    def fetchMore(self, parent):
        if parent.isValid():
            return

        itemsLeft = self.allJumpsCount - self.currentJumpCount
        newItems = min([self.BATCH_SIZE, itemsLeft])

        if newItems <= 0:
            return

        self.beginInsertRows(QtCore.QModelIndex(), self.currentJumpCount, self.currentJumpCount + newItems - 1)
        self.loadData()
        self.currentJumpCount += newItems
        self.endInsertRows()

    def loadData(self):
        result = self.reader.loadEntries(self.currentJumpCount, self.BATCH_SIZE)
        for entry in result:
            self._data.append(entry)

    def getSystemId(self, index):
        return self._data[index.row()]['id']

    def getStarIcon(self, starType):
        if starType in STAR_ICON_MAP:
            return STAR_ICON_MAP[starType] + "_small.png"
        elif starType is not None:
            return starType[0].lower() + "_small.png"
