import locale
from PyQt5.QtCore import QCoreApplication, QSettings, QPoint
from PyQt5 import QtWidgets, uic, QtCore
from PyQt5.QtCore import *
from package.Observer.FileWatcher import FileWatcher
from package.Observer.SignalHandler import SignalHandler as ObserverSignalHandler
from package.Config.ConfigManager import ConfigManager
from package.Database.DbFactory import DbFactory
from package.Ui.Handler.ObserverHandler import ObserverHandler as UiObserverHandler
from package.Observer.CalculationHandler import CalculationHandler
from package.Ui.Windows.FlightStatisticWindow import FlightStatisticWindow
from package.Ui.Windows.WorthScanningRulesWindow import WorthScanningRulesWindow
from package.Ui.Models.TableFlightLog import TableFlightLog
from package.WorthScanning.Scanner import Scanner
from package.Ui.Handler.DiscoveryHandler import DiscoveryHandler
from package.Ui.Windows.SettingsWindow import SettingsWindow
from package.Ui.Windows.ManageCheckerWindow import ManageCheckerWindow
from package.Ui.Windows.DbManagementWindow import DbManagementWindow
from package.Ui.Windows.StatisticWindow import StatisticWindow
from package.Ui.Windows.SearchWindow import SearchWindow
from package.Ui.Windows.SpecialSearchWindow import SpecialSearchWindow
from package.Ui.Actions import SystemAction


class MainUi(QtWidgets.QMainWindow):
    def __init__(self):
        super(MainUi, self).__init__()

        self.aboutWidget = None
        self.worthScanningRulesWindow = None
        self.settingsWindow = None
        self.checkerWindow = None
        self.dbManagementWindow = None
        self.statisticWindow = None
        self.searchWindow = None
        self.observerThread = None
        self.fileWatcher = None
        self.observerSignalHandler = None
        self.uiObserverHandler = UiObserverHandler(self)
        self.worthScanningModel = None
        self.worthScanningSortProxy = QtCore.QSortFilterProxyModel()
        self.flightStatisticWindow = FlightStatisticWindow()
        self.flightLogModel = None
        self.discoveryHandler = DiscoveryHandler(self)
        self.specialSearchWindow = None

        locale.setlocale(locale.LC_ALL, '')

        self.trans = QtCore.QTranslator(self)
        language = ConfigManager.getConfigValue("ui_language")
        self.trans.load("main_" + language + ".qm", "translations")
        QtWidgets.QApplication.instance().installTranslator(self.trans)

        self.transMisc = QtCore.QTranslator(self)
        self.transMisc.load("misc_" + language + ".qm", "translations")
        QtWidgets.QApplication.instance().installTranslator(self.transMisc)

        uic.loadUi('gui/gui.ui', self)
        self.installEventFilter(self)

        self.mainTab.tabBarClicked.connect(self.tabBarClicked)

        # restore window position
        self.settings = QSettings('timeagent', 'eliza')
        self.move(self.settings.value("pos", QPoint(50, 50)))

    def eventFilter(self, a0: 'QObject', a1: 'QEvent') -> bool:
        """
        obviously this is necessary to catch the close event (x button) when creating the ui with uic
        :param a0:
        :param a1:
        :return: bool
        """
        if a1.type() == QEvent.Close:
            self.settings.setValue("pos", self.pos())
            return True
        else:
            return super().eventFilter(a0, a1)

    def tabBarClicked(self, index) -> None:
        """
        reload flight log if tab is clicked
        :param index:
        :return: None
        """
        if index == 1:
            self.initFlightLog()

    def init(self) -> None:
        """
        init features in the main window
        :return: None
        """
        self.initFlightLog()
        self.initDiscovery()
        self.initMenu()

    def initMenu(self) ->None:
        """
        connect items in the menu
        :return: None
        """
        self.actionAbout.triggered.connect(self.aboutAction)
        self.actionWorthScanningRules.triggered.connect(self.openWorthScanningRulesWindow)
        self.actionSettings.triggered.connect(self.openSettingsWindow)
        self.actionManageChecker.triggered.connect(self.openCheckerWindow)
        self.actionDbManagement.triggered.connect(self.openDbManagementWindow)
        self.actionStatistik.triggered.connect(self.openStatisticWindow)
        self.actionSearch.triggered.connect(self.openSearchWindow)
        self.actionSpecialSearch.triggered.connect(self.openSpecialSearchWindow)

    #
    # menu actions
    #
    def aboutAction(self) -> None:
        """
        about message box
        :return: None
        """
        self.aboutWidget = QtWidgets.QMessageBox()
        self.aboutWidget.setIcon(QtWidgets.QMessageBox.Information)
        self.aboutWidget.setText(QCoreApplication.translate("MainUi", "about_text"))
        self.aboutWidget.setWindowTitle(QCoreApplication.translate("MainWindow", "About"))
        self.aboutWidget.setStandardButtons(QtWidgets.QMessageBox.Ok)
        self.aboutWidget.exec_()

    def openSettingsWindow(self) -> None:
        """
        Initialises and opens the settings window
        :return: None
        """
        if self.settingsWindow is None:
            self.settingsWindow = SettingsWindow()
            self.settingsWindow.setAttribute(QtCore.Qt.WA_DeleteOnClose)
            self.settingsWindow.closeSignal.connect(self.closeSettingsWindow)
            self.settingsWindow.eventSignal.connect(self.generalWindowHandler)
            self.settingsWindow.initSettings()
            self.settingsWindow.show()

    def openCheckerWindow(self) -> None:
        """
        Initialises and opens the window to manage checker
        :return: None
        """
        if self.checkerWindow is None:
            self.checkerWindow = ManageCheckerWindow()
            self.checkerWindow.setAttribute(QtCore.Qt.WA_DeleteOnClose)
            self.checkerWindow.closeSignal.connect(self.closeCheckerWindow)
            self.checkerWindow.init()
            self.checkerWindow.show()

    def openWorthScanningRulesWindow(self) -> None:
        """
        Initialises and opens the worth scanning rules window
        :return: None
        """
        if self.worthScanningRulesWindow is None:
            self.worthScanningRulesWindow = WorthScanningRulesWindow()
            self.worthScanningRulesWindow.setAttribute(QtCore.Qt.WA_DeleteOnClose)
            self.worthScanningRulesWindow.closeSignal.connect(self.updateScanner)
            self.worthScanningRulesWindow.init()
            self.worthScanningRulesWindow.show()

    def openDbManagementWindow(self) -> None:
        """
        Initialises and opens the db management window
        :return: None
        """
        if self.dbManagementWindow is None:
            self.dbManagementWindow = DbManagementWindow()
            self.dbManagementWindow.closeSignal.connect(self.closeDbManagementWindow)
            self.dbManagementWindow.eventSignal.connect(self.generalWindowHandler)
            self.dbManagementWindow.init()
            self.dbManagementWindow.show()

    def openStatisticWindow(self) -> None:
        """
        Initialises and opens the statistic window
        :return: None
        """
        if self.statisticWindow is None:
            self.statisticWindow = StatisticWindow()
            self.statisticWindow.closeSignal.connect(self.closeStatisticWindow)
            self.statisticWindow.openSystemWindowSignal.connect(self.openSystemViewFromStat)
            self.statisticWindow.initStatistics()
            self.statisticWindow.show()

    def openSearchWindow(self, systemAddress: int = None, bodyId: int = None) -> None:
        """
        Initialises the search window. if the window is already opened, update date in window
        :param systemAddress: int
        :return: None
        """
        # should be None if not give, but it's not?
        if not systemAddress:
            systemAddress = None

        if self.searchWindow is None:
            self.searchWindow = SearchWindow()
            self.searchWindow.setSystemAddress(systemAddress)
            if bodyId is not None:
                self.searchWindow.setBodyId(bodyId)

            self.searchWindow.closeSignal.connect(self.closeSearchWindow)
            self.searchWindow.init()
            self.searchWindow.show()
        else:
            self.searchWindow.setSystemAddress(systemAddress)
            if bodyId is not None:
                self.searchWindow.setBodyId(bodyId)

            self.searchWindow.initSystem()

    def openSpecialSearchWindow(self) -> None:
        """
        initialise the special search window and open it
        :return: None
        """
        if self.specialSearchWindow is None:
            self.specialSearchWindow = SpecialSearchWindow()
            self.specialSearchWindow.closeSignal.connect(self.closeSpecialSearchWindow)
            self.specialSearchWindow.openSystemWindowSignal.connect(self.openSystemViewFromStat)
            self.specialSearchWindow.init()
            self.specialSearchWindow.show()

    def updateScanner(self) -> None:
        """
        close handler for the worth scanning window. re-initialises the scanner in the filewatcher if necessary
        :return:
        """
        if isinstance(self.fileWatcher, FileWatcher):
            self.fileWatcher.getScanner().init()

        self.worthScanningRulesWindow = None

    def generalWindowHandler(self, action: int) -> None:
        """
        handler for various events in the settings window
        :param action: Enum - see SystemAction
        :return: None
        """
        if action == SystemAction.RELOAD_GUI:
            self.reloadGui()

        if action == SystemAction.RESTART_DISCOVERY:
            self.restartDiscovery()

        if action == SystemAction.START_DISCOVERY:
            self.discoveryStart()

        if action == SystemAction.STOP_DISCOVERY:
            self.discoveryStop()

        if action == SystemAction.DISCOVERY_RESET_DATA:
            self.discoveryHandler.resetData()

        if action == SystemAction.UPDATE_HOME_SYSTEM:
            if self.observerSignalHandler is not None:
                posX = ConfigManager.getConfigValue('home_system_pos_x')
                posY = ConfigManager.getConfigValue('home_system_pos_y')
                posZ = ConfigManager.getConfigValue('home_system_pos_z')

                self.observerSignalHandler.setHomeSystem(posX, posZ, posY)

    def restartDiscovery(self) -> None:
        """
        Will restart the discovery observer
        :return: None
        """
        self.discoveryStop()
        self.discoveryStart()

    def initDiscovery(self) -> None:
        """
        Initialises the discovery-observer
        :return: None
        """
        # Buttons
        self.buttonShowFlightStatistic.clicked.connect(self.openFlightStatisticWindow)
        self.buttonResetCreditsValue.clicked.connect(self.clickResetCreditsCount)

        # worth scanning model
        self.tableViewDiscoveryWorthScanning.horizontalHeader().hide()
        self.tableViewDiscoveryWorthScanning.verticalHeader().hide()
        self.tableViewDiscoveryWorthScanning.setSortingEnabled(True)

        # worth scanning proxy
        self.worthScanningSortProxy.setFilterKeyColumn(0)

        # init scanning amounts
        db = DbFactory.getDatabase()

        calc = CalculationHandler.getInstance(db)
        totalValue = calc.getTotalValue()
        totalBioValue = calc.getBioTotalValue()
        self.lineEditDiscoverySessionValue.setText("0")
        self.lineEditDiscoveryTotalValue.setText('{:,}'.format(totalValue).replace(',', '.'))
        self.lineEditDiscoveryBioSession.setText("0")
        self.lineEditDiscoveryBioTotalValue.setText('{:,}'.format(totalBioValue).replace(',', '.'))

        # init home system
        self.labelDiscoveryHomeSystem.setText(ConfigManager.getConfigValue('home_system_name'))
        self.labelDiscoveryLY.setText(QCoreApplication.translate("MainWindow", "LY").format("-"))

        # automatically start discovery mode
        self.discoveryStart()

    def initFlightLog(self) -> None:
        """
        Init the flight log
        :return: None
        """
        self.flightLogModel = TableFlightLog()
        self.tableFlightlogView.setModel(self.flightLogModel)
        self.tableFlightlogView.doubleClicked.connect(self.openSystemView)
        self.tableFlightlogView.setShowGrid(False)

        # set column sizes
        header = self.tableFlightlogView.horizontalHeader()
        for x in range(len(header)):
            if x == 0:
                header.setSectionResizeMode(x, QtWidgets.QHeaderView.Stretch)
            else:
                header.setSectionResizeMode(x, QtWidgets.QHeaderView.ResizeToContents)

    def openSystemView(self, index) -> None:
        """
        Open the detail system view window
        :param index:
        :return: None
        """
        systemAddress = self.tableFlightlogView.model().getSystemId(index)
        self.openSearchWindow(systemAddress)

    def isDiscoveryRunning(self) -> bool:
        """
        Returns if the discovery-observer is running
        :return: bool
        """
        if isinstance(self.observerThread, QtCore.QThread):
            return True

        return False

    def discoveryStart(self) -> None:
        """
        Starts the discovery-observer
        :return: None
        """
        if self.uiObserverHandler.checkObserverSettings() is False:
            return

        if self.isDiscoveryRunning():
            return

        # create Thread
        self.observerThread = QtCore.QThread()

        # create signal handler
        self.observerSignalHandler = ObserverSignalHandler()
        self.observerSignalHandler.setView(self)

        # create and init worthscanning scanner
        scanner = Scanner()
        scanner.init()

        self.fileWatcher = FileWatcher()
        self.fileWatcher.fileWatcherSignal.connect(self.observerSignalHandler.parseSignal)
        self.fileWatcher.directDisplaySignal.connect(self.displayDiscoveryText)
        self.fileWatcher.setScanner(scanner)
        self.fileWatcher.setupLineHandler()
        self.fileWatcher.start()
        self.fileWatcher.moveToThread(self.observerThread)
        self.observerThread.started.connect(self.fileWatcher.watcher)
        self.observerThread.start()

    def discoveryStop(self) -> None:
        """
        Stops the discovery-observer
        :return: None
        """
        if not self.isDiscoveryRunning():
            return

        if isinstance(self.fileWatcher, FileWatcher):
            self.fileWatcher.stop()

        self.observerThread.quit()
        self.observerThread.wait()
        self.observerThread = None

    def displayDiscoveryText(self, message: str) -> None:
        """
        signal handler to display messages from the observer in the textbox
        :param message: string
        :return: None
        """
        self.textBrowserDiscovery.append(message)

    def reloadGui(self) -> None:
        """
        Reload the GUI to apply language changes
        :return: None
        """
        language = ConfigManager.getConfigValue("ui_language")
        QtWidgets.QApplication.instance().removeTranslator(self.trans)
        languageFile = "main_" + language + ".qm"
        self.trans.load(languageFile, "translations")
        QtWidgets.QApplication.instance().installTranslator(self.trans)
        uic.loadUi('gui/gui.ui', self)
        self.init()

    def openFlightStatisticWindow(self) -> None:
        """
        Open the flight statistic window
        :return: None
        """
        self.flightStatisticWindow.init()
        self.flightStatisticWindow.show()

    def closeEvent(self, event) -> None:
        """
        stop discovery-observer if app is closed
        :param event:
        :return: None
        """
        self.discoveryStop()

    def clickResetCreditsCount(self) -> None:
        """
        Handler for the reset credits button
        :return: None
        """
        self.discoveryHandler.resetCreditsCount()

    #
    # Close window handlers
    #
    def closeSettingsWindow(self) -> None:
        """
        Removes the settings window instance
        :return:
        """
        self.settingsWindow = None

    def closeCheckerWindow(self) -> None:
        """
        remove the instance of the window to manage checkers
        :return:
        """
        self.checkerWindow = None

    def closeStatisticWindow(self) -> None:
        """
        remove the statistic window instance
        :return:
        """
        self.statisticWindow = None

    def closeSearchWindow(self) -> None:
        """
        remove the search window instance
        :return:
        """
        self.searchWindow = None

    def closeDbManagementWindow(self) -> None:
        """
        remove the db management window instance
        :return:
        """
        self.dbManagementWindow = None

    def openSystemViewFromStat(self, object) -> None:
        """
        Helper to open the search value based on the data coming from the statistic table
        :param object: dict
        :return: None
        """
        bodyId = None
        if 'meta_body_id' in object:
            bodyId = object['meta_body_id']

        self.openSearchWindow(object['meta_system_address'], bodyId)

    def closeSpecialSearchWindow(self) -> None:
        """
        remove the special search window instance
        :return:
        """
        self.specialSearchWindow = None