import locale
import urllib
from PyQt5.QtWidgets import *
from PyQt5.QtGui import *
from PyQt5.QtCore import *
from package.Helper.UiHelper import *
from package.Ui.Element.MiniMap import MiniMap
from package.Elite.System import System


class SystemInfo(QFrame):
    def __init__(self, parent=None, system: System = None):
        super().__init__(parent)
        self.system = system

        self.labelSystemName = None
        self.labelEdsmLink = None

        self.setFrameStyle(QFrame.Panel | QFrame.Raised)
        self.setLineWidth(2)

        self.init()

    def init(self) -> None:
        # main layout
        mainLayout = QVBoxLayout(self)
        mainLayout.setContentsMargins(6, 3, 6, 3)

        # horizontal layout for the content
        hlayout = QHBoxLayout()
        hlayout.setContentsMargins(0, 0, 0, 0)

        # system map
        playerPos = self.system.getPlayerPosition()
        minimap = MiniMap(
            self.system.getPosX(),
            self.system.getPosY(),
            playerPos[0],
            playerPos[2]
        )
        hlayout.addWidget(minimap)

        # vertical layout for other data
        contentLayout = QVBoxLayout()
        contentLayout.setContentsMargins(10, 10, 0, 0)

        # system name
        systemNameLayout = QHBoxLayout()
        systemNameLayout.setContentsMargins(0, 0, 0, 20)
        systemNameLayout.addWidget(getDescLabel(
            QCoreApplication.translate("Systemview", "system")
        ))
        self.labelSystemName = getContentLabel(self.system.getName())
        systemNameLayout.addWidget(self.labelSystemName)
        contentLayout.addLayout(systemNameLayout)

        # grid layout for data
        dataLayout = QGridLayout()

        # position
        dataLayout.addWidget(
            getDescLabel(QCoreApplication.translate("Systemview", "coordinates")),
            0,
            0
        )
        dataLayout.addWidget(
            getContentLabel(
                locale.format_string('%.2f', self.system.getPosX(), True) + " / " +
                locale.format_string('%.2f', self.system.getPosY(), True) + " / " +
                locale.format_string('%.2f', self.system.getPosZ(), True)
            ),
            0,
            1
        )

        # distance
        dataLayout.addWidget(
            getDescLabel(QCoreApplication.translate("Systemview", "distance")),
            1,
            0
        )
        dataLayout.addWidget(
            getContentLabel(
                locale.format_string('%.2f', self.system.getDistance(), True) + " " +
                QCoreApplication.translate("Systemview", "lightsecond_abbr")
            ),
            1,
            1
        )

        # body counts
        dataLayout.addWidget(
            getDescLabel(QCoreApplication.translate("Systemview", "bodies")),
            2,
            0
        )
        dataLayout.addWidget(
            getContentLabel(
                str(self.system.getStarCount()) + ' ' +
                QCoreApplication.translate("Systemview", "stars") + ', ' +
                str(self.system.getPlanetCount()) + ' ' +
                QCoreApplication.translate("Systemview", "planets")
            ),
            2,
            1
        )

        # scan complete
        dataLayout.addWidget(
            getDescLabel(QCoreApplication.translate("Systemview", "scan_complete")),
            3,
            0
        )

        if self.system.isScanComplete() == 1:
            completeText = QCoreApplication.translate("Flightlog", "yes")
        else:
            completeText = QCoreApplication.translate("Flightlog", "no")

        dataLayout.addWidget(
            getContentLabel(
                completeText
            ),
            3,
            1
        )

        # value
        dataLayout.addWidget(
            getDescLabel(QCoreApplication.translate("Systemview", "estimated_value")),
            5,
            0
        )
        dataLayout.addWidget(
            getContentLabel(
                locale.format_string('%.0f', self.system.getEstimatedValue(), True) + ' ' +
                QCoreApplication.translate("Systemview", "credits")
            ),
            5,
            1
        )

        contentLayout.addLayout(dataLayout)

        # edsm link
        edsmLink = "https://www.edsm.net/de/system/id//name?systemName=" + urllib.parse.quote_plus(self.system.getName())
        edsmText = '<a href="' + edsmLink + '">' + QCoreApplication.translate("Systemview", "edsm_search") + "</a>"

        self.labelEdsmLink = getDescLabel(edsmText)
        self.labelEdsmLink.setOpenExternalLinks(True)
        self.labelEdsmLink.setTextInteractionFlags(Qt.LinksAccessibleByMouse)
        self.labelEdsmLink.setContentsMargins(0, 20, 0, 20)
        contentLayout.addWidget(self.labelEdsmLink)

        contentLayout.addStretch(1)

        # add contentLayout to hlayout
        hlayout.addLayout(contentLayout)

        # additional layout for notable object counts
        additionalLayout = QVBoxLayout()
        additionalLayout.setContentsMargins(10, 10, 0, 0)

        labelNotableObjects = getDescLabel(
            QCoreApplication.translate("Systemview", "notable_objects")
        )
        labelNotableObjects.setContentsMargins(0, 0, 0, 20)
        additionalLayout.addWidget(labelNotableObjects)

        if not self.system.hasNotableObjects() and len(self.system.getStellarPhenomena()) == 0:
            label = getContentLabel(
                QCoreApplication.translate("Systemview", "no_notable_objects")
            )
            additionalLayout.addWidget(label)
        else:
            # grid layout for data
            dataLayout = QGridLayout()

            # terraformer
            dataLayout.addWidget(
                getDescLabel(QCoreApplication.translate("Systemview", "terraformable")),
                0,
                0
            )
            dataLayout.addWidget(
                getContentLabel(
                    str(self.system.getTerraformableCount())
                ),
                0,
                1
            )

            # water worlds
            dataLayout.addWidget(
                getDescLabel(QCoreApplication.translate("Systemview", "waterworlds")),
                1,
                0
            )
            dataLayout.addWidget(
                getContentLabel(
                    str(self.system.getWaterWorldCount())
                ),
                1,
                1
            )

            # earth likes
            dataLayout.addWidget(
                getDescLabel(QCoreApplication.translate("Systemview", "earthlikes")),
                2,
                0
            )
            dataLayout.addWidget(
                getContentLabel(
                    str(self.system.getEartlikesCount())
                ),
                2,
                1
            )

            # ammonia worlds
            dataLayout.addWidget(
                getDescLabel(QCoreApplication.translate("Systemview", "ammoniaworlds")),
                3,
                0
            )
            dataLayout.addWidget(
                getContentLabel(
                    str(self.system.getAmmoniaWorldCount())
                ),
                3,
                1
            )

            additionalLayout.addLayout(dataLayout)

            if len(self.system.getStellarPhenomena()) > 0:
                additionalLayout.addLayout(self.createStellarPhenomenaBox())

        additionalLayout.addStretch(1)
        hlayout.addLayout(additionalLayout)

        # add hlayout to main
        mainLayout.addLayout(hlayout)

        self.setLayout(mainLayout)
        self.setSizePolicy(QSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum))

    def createStellarPhenomenaBox(self):
        layout = QVBoxLayout()

        for name in self.system.getStellarPhenomena():
            layout.addWidget(getContentLabel(name))

        return layout
