from PyQt5.QtWidgets import *
from PyQt5.QtGui import *
from PyQt5.QtCore import *
from package.Helper.UiHelper import *
from package.Journal.Events.Star import Star
from package.Ui.Element.RingInfo import RingInfo


class StarInfo(QFrame):
    def __init__(self, parent=None, body: Star = None):
        super().__init__(parent)
        self.body = body

        self.setFrameStyle(QFrame.Panel | QFrame.Raised)
        self.setLineWidth(2)

        self.init()

    def init(self) -> None:
        # main layout
        mainLayout = QVBoxLayout(self)
        mainLayout.setContentsMargins(6, 3, 6, 3)

        mainLayout.addWidget(self.createStarNameFrame())

        hlay = QHBoxLayout()
        hlay.addWidget(self.createBasicDataBox())
        hlay.addWidget(self.createOrbitalDataBox())
        mainLayout.addLayout(hlay)

        if len(self.body.getRings()) > 0:
            mainLayout.addWidget(self.createRingsBox())

    def createStarNameFrame(self) -> QFrame:
        nameTopFrame = QFrame()
        nameTopFrame.setFrameStyle(QFrame.Panel | QFrame.Raised)
        nameTopFrame.setLineWidth(1)

        # star name and type
        topFrameLayout = QVBoxLayout(nameTopFrame)

        nameFrameLayout = QHBoxLayout()
        nameLabel = getDescLabel(self.body.getName())

        boldFont = QFont()
        boldFont.setBold(True)
        nameLabel.setFont(boldFont)
        nameFrameLayout.addWidget(nameLabel)

        typeLabel = getDescLabel(
            "(" + QCoreApplication.translate("BodyTypes", self.body.getType()) + ")"
        )

        sizePolicy = QSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
        sizePolicy.setHorizontalStretch(1)
        typeLabel.setSizePolicy(sizePolicy)
        nameFrameLayout.addWidget(typeLabel)

        topFrameLayout.addLayout(nameFrameLayout)

        # discovery date
        topFrameLayout.addLayout(self.createDiscoveryDate())

        # first discovered icon
        if self.body.wasDiscovered() is not True:
            # horizontal layout for icon
            iconLayout = QHBoxLayout()

            picLabel = QLabel()
            pixmap = QPixmap("gui/images/icons/firstdiscovered_small.png")
            picLabel.setPixmap(pixmap)
            picLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
            picLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            picLabel.setToolTip(QCoreApplication.translate("Systemview", "first_discovered"))
            iconLayout.addWidget(picLabel)

            descLabel = getDescLabel(QCoreApplication.translate("Systemview", "first_discovered"))
            descLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
            descLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            descLabel.setContentsMargins(0, 0, 15, 0)
            iconLayout.addWidget(descLabel)

            iconLayout.addStretch(1)
            topFrameLayout.addLayout(iconLayout)

        return nameTopFrame

    def createOrbitalDataBox(self):
        box = QGroupBox(QCoreApplication.translate("Systemview", "orbital_data"))

        dataLayout = QGridLayout()
        # distance
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "distance")
            ),
            0,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getDistanceAsString()),
            0,
            1
        )

        # orbital parameter
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "semi_major_axis")
            ),
            1,
            0        )

        dataLayout.addWidget(
            getContentLabel(self.body.getSemiMajorAxisAsString()),
            1,
            1
        )

        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "eccentricity")
            ),
            2,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getEccentricityAsString()),
            2,
            1
        )

        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "inclination")
            ),
            3,
            0
        )

        dataLayout.addWidget(
            getContentLabel(str(self.body.getInclinationAsString())),
            3,
            1
        )

        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "periapsis")
            ),
            4,
            0
        )

        dataLayout.addWidget(
            getContentLabel(str(self.body.getPeriapsis())),
            4,
            1
        )

        # orbital period
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "orbital_period")
            ),
            5,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getOrbitalPeriodAsString()),
            5,
            1
        )

        box.setLayout(dataLayout)
        return box

    def createBasicDataBox(self):
        box = QGroupBox(QCoreApplication.translate("Systemview", "planet"))

        dataLayout = QGridLayout()

        # age
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "age")
            ),
            0,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getAgeAsString()),
            0,
            1
        )

        # mass
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "solar_mass")
            ),
            1,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getMassAsString()),
            1,
            1
        )

        # radius
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "solar_radii")
            ),
            2,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getRadiusAsString()),
            2,
            1
        )

        # temperature
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "temperature")
            ),
            3,
            0
        )

        tempTxt = self.body.getTemperatureAsKelvin() + ' (' + self.body.getTemperatureAsCelsius() + ')'

        dataLayout.addWidget(
            getContentLabel(tempTxt),
            3,
            1
        )

        # spectral class
        spectralClass = self.body.getSpectralClass()

        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "spectral_class")
            ),
            4,
            0
        )

        dataLayout.addWidget(
            getContentLabel(spectralClass),
            4,
            1
        )

        # rotation period
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "rotation_period")
            ),
            5,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getRotationPeriodAsString()),
            5,
            1
        )

        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "magnitude")
            ),
            6,
            0
        )

        dataLayout.addWidget(
            getContentLabel(str(self.body.getAbsoluteMagnitude())),
            6,
            1
        )

        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "axial_tilt")
            ),
            7,
            0
        )

        dataLayout.addWidget(
            getContentLabel(str(self.body.getAxialTilt())),
            7,
            1
        )

        box.setLayout(dataLayout)
        return box

    def createRingsBox(self):
        box = QGroupBox(QCoreApplication.translate("Systemview", "rings"))
        hlayout = QHBoxLayout()

        for ring in self.body.getRings():
            ringName = ring.getName().replace(self.body.getName(), '').strip()
            ringInfo = RingInfo(ringName, ring)
            hlayout.addWidget(ringInfo)

        box.setLayout(hlayout)
        return box

    def createDiscoveryDate(self):
        discoveryLayout = QHBoxLayout()
        discoveryLayout.setContentsMargins(0, 5, 0, 5)
        discoveredDate = self.body.getFormatedDiscoveryDate()

        discoveryTextLabel = getDescLabel(
            text=QCoreApplication.translate("Systemview", "discovered_date"),
            small=True
        )
        discoveryLayout.addWidget(discoveryTextLabel)

        discoveryLayout.addWidget(
            getContentLabel(text=discoveredDate, small=True)
        )

        return discoveryLayout