from PyQt5.QtWidgets import *
from PyQt5.QtGui import *
from PyQt5.QtCore import *
from package.Helper.UiHelper import *
from package.Journal.Events.Planet import Planet
from package.Ui.Element.RingInfo import RingInfo
from package.Ui.Element.BioBox import BioBox
from package.Ui.Element.GeoBox import GeoBox
from package.Ui.Element.MiscCodexBox import MiscCodexBox
from package.Ui.Element.AtmosphereBox import AtmosphereBox
from package.Journal.Events.ScanOrganic import ScanOrganic
from package.Journal.Events.SSASignal import SSASignal
from package.Helper.Apsis import Apsis


class PlanetInfo(QFrame):
    def __init__(self, parent=None, body: Planet = None):
        super().__init__(parent)
        self.body = body

        self.setFrameStyle(QFrame.Panel | QFrame.Raised)
        self.setLineWidth(2)
        self.setContentsMargins(0, 0, 0, 20)

        self.init()

    def init(self) -> None:
        # main layout
        mainLayout = QVBoxLayout(self)
        mainLayout.setContentsMargins(6, 3, 6, 3)

        mainLayout.addWidget(self.createPlanetNameFrame())

        hlay = QHBoxLayout()
        hlay.addWidget(self.createOrbitalDataBox())
        hlay.addWidget(self.createBasicDataBox())
        mainLayout.addLayout(hlay)

        if len(self.body.getOrganicScans(ScanOrganic.SCAN_TYPE_LOG)) > 0 or\
                len(self.body.getOrganicScans(ScanOrganic.SCAN_TYPE_ANALYSE)) > 0:
            mainLayout.addWidget(BioBox(self.body))

        anotherHLayout = QHBoxLayout()

        atmosphereColumns = 5
        if self.body.getGeoSignalCount() > 0:
            atmosphereColumns = 3

        if len(self.body.getAtmosphereComposition()) > 0:
            anotherHLayout.addWidget(AtmosphereBox(self.body, atmosphereColumns))

        if self.body.getGeoSignalCount() > 0:
            anotherHLayout.addWidget(GeoBox(self.body))

        if len(self.body.getCodexEntries(SSASignal.DB_TYPE_MISC)) > 0:
            anotherHLayout.addWidget(MiscCodexBox(self.body))

        mainLayout.addLayout(anotherHLayout)

        if len(self.body.getRings()) > 0:
            mainLayout.addWidget(self.createRingsBox())

        if len(self.body.getMaterial()) > 0:
            mainLayout.addWidget(self.createMaterialsBox())

    def createPlanetNameFrame(self) -> QFrame:
        nameTopFrame = QFrame()
        nameTopFrame.setFrameStyle(QFrame.Panel | QFrame.Raised)
        nameTopFrame.setLineWidth(1)

        topFrameLayout = QVBoxLayout(nameTopFrame)

        # planet name and type
        nameFrameLayout = QHBoxLayout()
        nameLabel = getDescLabel(text=self.body.getName(), bold=True)
        nameFrameLayout.addWidget(nameLabel)

        typeLabel = getDescLabel(
            "(" + QCoreApplication.translate("BodyTypes", self.body.getType()) + ")"
        )

        sizePolicy = QSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
        sizePolicy.setHorizontalStretch(1)
        typeLabel.setSizePolicy(sizePolicy)

        nameFrameLayout.addWidget(typeLabel)
        topFrameLayout.addLayout(nameFrameLayout)

        # discovery date
        topFrameLayout.addLayout(self.createDiscoveryDate())

        # horizontal layout for icons
        iconLayout = QHBoxLayout()

        numIconsSet = 0

        # first discovered icon
        if self.body.wasDiscovered() is not True:
            picLabel = QLabel()
            pixmap = QPixmap("gui/images/icons/firstdiscovered_small.png")
            picLabel.setPixmap(pixmap)
            picLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
            picLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            picLabel.setToolTip(QCoreApplication.translate("Systemview", "first_discovered"))
            iconLayout.addWidget(picLabel)

            descLabel = getDescLabel(QCoreApplication.translate("Systemview", "first_discovered"))
            descLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
            descLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            descLabel.setContentsMargins(0, 0, 15, 0)
            iconLayout.addWidget(descLabel)
            numIconsSet += 1

        # detail scan icon
        if self.body.isDetailScaned():
            picLabel = QLabel()
            pixmap = QPixmap("gui/images/icons/mapped_small.png")
            picLabel.setPixmap(pixmap)
            picLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
            picLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            picLabel.setToolTip(QCoreApplication.translate("Systemview", "detail_scan"))
            iconLayout.addWidget(picLabel)

            descLabel = getDescLabel(QCoreApplication.translate("Systemview", "detail_scan"))
            descLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
            descLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            descLabel.setContentsMargins(0, 0, 15, 0)
            iconLayout.addWidget(descLabel)
            numIconsSet += 1

        # tidal lock icon
        if self.body.isTidalLocked():
            picLabel = QLabel()
            pixmap = QPixmap("gui/images/icons/tidal_lock_small.png")
            picLabel.setPixmap(pixmap)
            picLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
            picLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            picLabel.setToolTip(QCoreApplication.translate("Systemview", "tidal_lock"))
            iconLayout.addWidget(picLabel)

            descLabel = getDescLabel(QCoreApplication.translate("Systemview", "tidal_lock"))
            descLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
            descLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            descLabel.setContentsMargins(0, 0, 15, 0)
            iconLayout.addWidget(descLabel)
            numIconsSet += 1

        # terraformable icon
        if self.body.isTerraformable():
            picLabel = QLabel()
            pixmap = QPixmap("gui/images/icons/terraformable_small.png")
            picLabel.setPixmap(pixmap)
            picLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
            picLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            picLabel.setToolTip(QCoreApplication.translate("Systemview", "tooltip_terraformable"))
            iconLayout.addWidget(picLabel)

            descLabel = getDescLabel(QCoreApplication.translate("Systemview", "tooltip_terraformable"))
            descLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
            descLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            descLabel.setContentsMargins(0, 0, 15, 0)
            iconLayout.addWidget(descLabel)
            numIconsSet += 1

        # binary icon
        if self.body.isBinaryObject():
            picLabel = QLabel()
            pixmap = QPixmap("gui/images/icons/binary_small.png")
            picLabel.setPixmap(pixmap)
            picLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
            picLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            picLabel.setToolTip(QCoreApplication.translate("Systemview", "binary_orbit"))
            iconLayout.addWidget(picLabel)

            descLabel = getDescLabel(QCoreApplication.translate("Systemview", "binary_orbit"))
            descLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
            descLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            descLabel.setContentsMargins(0, 0, 15, 0)
            iconLayout.addWidget(descLabel)
            numIconsSet += 1

        # landable icon
        if self.body.isLandable():
            picLabel = QLabel()
            pixmap = QPixmap("gui/images/icons/landable_small.png")
            picLabel.setPixmap(pixmap)
            picLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
            picLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            picLabel.setToolTip(QCoreApplication.translate("Systemview", "tooltip_landable"))
            iconLayout.addWidget(picLabel)

            descLabel = getDescLabel(QCoreApplication.translate("Systemview", "tooltip_landable"))
            descLabel.setAlignment(Qt.AlignTop | Qt.AlignLeft)
            descLabel.setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            descLabel.setContentsMargins(0, 0, 15, 0)
            iconLayout.addWidget(descLabel)
            numIconsSet += 1

        # add spacer if more than one icon is displayed
        if numIconsSet > 1:
            iconLayout.addStretch(1)

        topFrameLayout.addLayout(iconLayout)

        return nameTopFrame

    def createDiscoveryDate(self):
        discoveryLayout = QHBoxLayout()
        discoveryLayout.setContentsMargins(0, 5, 0, 5)
        discoveredDate = self.body.getFormatedDiscoveryDate()

        discoveryTextLabel = getDescLabel(
            text=QCoreApplication.translate("Systemview", "discovered_date"),
            small=True
        )
        discoveryLayout.addWidget(discoveryTextLabel)

        discoveryLayout.addWidget(
            getContentLabel(text=discoveredDate, small=True)
        )

        return discoveryLayout

    def createOrbitalDataBox(self):
        box = QGroupBox(QCoreApplication.translate("Systemview", "orbital_data"))

        # perihel / aphel
        rowAdd = 0
        perihel = Apsis.calculatePerihelion(
            self.body.getSemiMajorAxis(),
            self.body.getEccentricity()
        )
        aphel = Apsis.calculateAphelion(
            self.body.getSemiMajorAxis(),
            self.body.getEccentricity()
        )

        dataLayout = QGridLayout()
        #distance
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "distance")
            ),
            0,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getDistanceAsString(2)),
            0,
            1
        )

        # orbital parameter
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "semi_major_axis")
            ),
            1,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getSemiMajorAxisAsString()),
            1,
            1
        )

        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "eccentricity")
            ),
            2,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getEccentricityAsString()),
            2,
            1
        )

        if self.body.getEccentricity() > 0.9:
            rowAdd = 2

            perihelText = QCoreApplication.translate("Systemview", "perihel_with_value").format(perihel)
            aphelText = QCoreApplication.translate("Systemview", "aphel_with_value").format(aphel)

            dataLayout.addWidget(
                getContentLabel(
                    perihelText
                ),
                3,
                1
            )

            dataLayout.addWidget(
                getContentLabel(
                    aphelText
                ),
                4,
                1
            )

        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "inclination")
            ),
            3 + rowAdd,
            0
        )

        dataLayout.addWidget(
            getContentLabel(str(self.body.getInclinationAsString())),
            3 + rowAdd,
            1
        )

        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "periapsis")
            ),
            4 + rowAdd,
            0
        )

        dataLayout.addWidget(
            getContentLabel(str(self.body.getPeriapsis())),
            4 + rowAdd,
            1
        )

        # orbital period
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "orbital_period")
            ),
            5 + rowAdd,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getOrbitalPeriodAsString()),
            5 + rowAdd,
            1
        )

        box.setLayout(dataLayout)
        return box

    def createBasicDataBox(self):
        box = QGroupBox(QCoreApplication.translate("Systemview", "planet"))

        dataLayout = QGridLayout()

        # gravity
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "gravity")
            ),
            0,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getGravityAsString()),
            0,
            1
        )

        # mass
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "earth_mass")
            ),
            1,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getMassAsString()),
            1,
            1
        )

        # radius
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "body_radii")
            ),
            2,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getRadiusAsString()),
            2,
            1
        )

        # temperature
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "temperature")
            ),
            3,
            0
        )

        tempTxt = self.body.getTemperatureAsKelvin() + ' (' + self.body.getTemperatureAsCelsius() + ')'

        dataLayout.addWidget(
            getContentLabel(tempTxt),
            3,
            1
        )

        # rotation period
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "rotation_period")
            ),
            4,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getRotationPeriodAsString()),
            4,
            1
        )

        # axial tilt
        dataLayout.addWidget(
            getDescLabel(
                QCoreApplication.translate("Systemview", "axial_tilt")
            ),
            5,
            0
        )

        dataLayout.addWidget(
            getContentLabel(self.body.getAxialTiltAsString()),
            5,
            1
        )

        if self.body.getVolcanism() is not None and self.body.getVolcanism() != "":
            # volcanism
            dataLayout.addWidget(
                getDescLabel(
                    QCoreApplication.translate("Systemview", "volcanism")
                ),
                6,
                0
            )

            dataLayout.addWidget(
                getContentLabel(self.body.getVolcanism()),
                6,
                1
            )

        box.setLayout(dataLayout)
        return box

    def createMaterialsBox(self):
        box = QGroupBox(QCoreApplication.translate("Systemview", "materials"))

        dataLayout = QGridLayout()

        row = 0
        column = 0
        for mat in self.body.getMaterial():
            dataLayout.addWidget(
                getDescLabel(mat.getNameLocalised() + ":"),
                row,
                column
            )

            column += 1

            dataLayout.addWidget(
                getContentLabel(mat.getPercentAsString()),
                row,
                column
            )

            column += 1

            if column > 7:
                column = 0
                row += 1

        box.setLayout(dataLayout)
        return box

    def createRingsBox(self):
        box = QGroupBox(QCoreApplication.translate("Systemview", "rings"))
        hlayout = QHBoxLayout()

        for ring in self.body.getRings():
            ringName = ring.getName().replace(self.body.getName(), '').strip()
            ringInfo = RingInfo(ringName, ring)
            hlayout.addWidget(ringInfo)

        box.setLayout(hlayout)
        return box
