from functools import partial
from PyQt5 import QtWidgets, QtCore
from package.Helper.ClassLoader import ClassLoader
from package.Database.DbFactory import DbFactory

class CheckerEntry(QtWidgets.QWidget):
    checkerChangedSignal = QtCore.pyqtSignal()
    checkerConfigButtonSignal = QtCore.pyqtSignal(str)

    def __init__(self, checkerId: int, checkerName: str, active: bool):
        super().__init__()
        self.checkerName = checkerName
        self.checkerId = checkerId
        self.active = active
        self.checkbox = None
        self.configButton = None
        self.createUi()

    def createUi(self) -> None:
        """
        Define UI elements
        :return: None
        """
        mainLayout = QtWidgets.QHBoxLayout()

        self.checkbox = QtWidgets.QCheckBox(self.checkerName)
        if self.active:
            self.checkbox.setChecked(True)

        self.checkbox.stateChanged.connect(partial(self.checkerChanged, self.checkerId))
        mainLayout.addWidget(self.checkbox)

        checkerClass = self.getCheckerClass(self.checkerName)
        if checkerClass.hasConfigValues():
            self.configButton = QtWidgets.QPushButton()
            self.configButton.setIcon(self.style().standardIcon(QtWidgets.QStyle.SP_FileDialogListView))
            self.configButton.setSizePolicy(QtWidgets.QSizePolicy.Minimum, QtWidgets.QSizePolicy.Minimum)
            self.configButton.clicked.connect(self.checkerConfigButtonClicked)

            mainLayout.addWidget(self.configButton)

            spacer = QtWidgets.QSpacerItem(20, 20, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Minimum)
            mainLayout.addItem(spacer)

        self.setLayout(mainLayout)

    def getCheckerClass(self, name: str):
        packageNameBase = "package.Observer.Checker."
        packageName = packageNameBase + name
        checkerClass = ClassLoader.getClass(packageName, name)

        return checkerClass

    def setChecked(self, value: bool) -> None:
        """
        Set the checkbox status. called from the window
        :param value:
        :return: None
        """
        self.checkbox.setChecked(value)

    def checkerChanged(self, id: int) -> None:
        """
        Updates the checker status in the db
        :param id: int. id of the changed checker
        :return: None
        """
        print("checker changed")
        state = 0
        if self.checkbox.isChecked():
            state = 1

        self.setCheckerState(id, state)

    def setCheckerState(self, id, checked):
        """
        Updates checker status in db
        :param id: id of the checker
        :param checked: 0 if checker is unchecked, 1 if it's checked
        :return: None
        """
        db = DbFactory.getDatabase()
        data = {"active": checked}
        where = {"id": id}
        db.update("configChecker", data, where)

        print("emit signal")
        self.checkerChangedSignal.emit()

    def checkerConfigButtonClicked(self) -> None:
        """
        Emit signal that the button to open the config window was clicked
        :return: None
        """
        self.checkerConfigButtonSignal.emit(self.checkerName)
