from PyQt5.QtCore import QCoreApplication
from package.Statistics.StatisticsData import StatisticsData
from package.Elite.StarTypes import StarTypes
from package.Elite.PlanetTypes import PlanetTypes
from package.Service.StarService import StarService
from package.Service.PlanetService import PlanetService


class StatisticsReader:
    def __init__(self, db):
        self.db = db
        self.starService = StarService()
        self.planetService = PlanetService()

    def getPlanetCriteria(self):
        return StatisticsData.PLANET_CRITERIA

    def getStarCriteria(self):
        return StatisticsData.STAR_CRITERIA

    def getStarTypes(self) -> dict:
        """
        Returns a dict of all star types for the statistic
        :return: dict
        """
        customTypes = {
            'TYPE_GROUP_CARBON': 'CS,C,CN,CJ,CH,CHd',
            'TYPE_GROUP_MS': 'MS,S',
            'TYPE_GROUP_WOLF_RAYET': 'W,WN,WNC,WC,WO',
            'TYPE_GROUP_WHITE_DWARF': 'D,DA,DAB,DAO,DAZ,DAV,DB,DBZ,DBV,DO,DOV,DQ,DC,DCV,DX',
            'TYPE_ALL': "--all--",
        }

        # this is a quite fast concat method for dicts
        return dict(StarTypes.TYPES, **customTypes)

    def getPlanetTypes(self) -> dict:
        """
        Returns a dict of all planet types for the statistic
        :return: dict
        """
        customTypes = {
            'TYPE_GROUP_WATER_GIANTS': "Water giant,Water giant with life",
            'TYPE_ALL': "--all--",
        }

        return dict(PlanetTypes.TYPES, **customTypes)

    def getSortOptions(self):
        return StatisticsData.SORT_OPTIONS

    def getKeyFor(self, listname, text):
        key = None
        for i in listname:
            if i["key"] == text:
                key = i["key"]

        return key

    def getPlanetStatistics(self, keyType, criteria, sort, count, own, landable, ringsOnly: bool=False):
        keyCriteria = self.getKeyFor(StatisticsData.PLANET_CRITERIA, criteria)
        keySort = self.getKeyFor(StatisticsData.SORT_OPTIONS, sort)

        if keyType != '--all--':
            if "," in keyType:
                typeValue = keyType.split(",")
            else:
                typeValue = [keyType]

        whereParts = []
        whereValues = []

        query = """SELECT b.*, (r.outer_rad - r.inner_rad) AS ring_width, r.outer_rad as ring_outer_rad
                    FROM bodies b
                    LEFT JOIN body_rings br ON br.body_id = b.id
                    LEFT JOIN rings r on br.ring_id = r.id
                """

        if keyType != '--all--':
            format_strings = ','.join(['%s'] * len(typeValue))
            whereParts.append("b.type IN (" + format_strings + ")")
            whereValues += tuple(typeValue)

        if own:
            whereParts.append("b.was_discovered = 0")

        if landable:
            whereParts.append("b.landable = 1")

        if ringsOnly:
            whereParts.append("br.body_id IS NOT NULL")

        # append where
        if len(whereParts) > 0:
            query += " WHERE "
            query += " AND ".join(whereParts)

        # append order
        query += " ORDER BY " + keyCriteria + " " + keySort

        # append limit
        if count is not None:
            query += " LIMIT " + str(count)

        result = self.db.selectPlain(query, whereValues)

        # only unique entries - this needs to be reworked... kinda hacky
        uniqueEntries = []
        tmpIds = []
        for entry in result:
            if entry['id'] not in tmpIds:
                tmpIds.append(entry['id'])
                uniqueEntries.append(entry)

        planets = self.planetService.getFromData(uniqueEntries)

        return self.generatePlanetData(planets, ringsOnly)

    def generatePlanetData(self, bodies, ringsOnly: bool):
        data = []
        for body in bodies:
            tmp = {
                QCoreApplication.translate("additional", "name"): body.getName(),
                QCoreApplication.translate("additional", "type"): body.getType(),
                QCoreApplication.translate("additional", "mass"): body.getMassAsString(),
                QCoreApplication.translate("additional", "radius"): body.getRadiusAsString(),
                QCoreApplication.translate("additional", "gravity"): body.getGravityAsString(),
                QCoreApplication.translate("additional", "temperature"): body.getTemperatureAsString(),
                QCoreApplication.translate("additional", "orbital_period"): body.getOrbitalPeriodAsString(),
                QCoreApplication.translate("additional", "rotation_period"): body.getRotationPeriodAsString()
            }

            if ringsOnly:
                tmp[QCoreApplication.translate("additional", "ring_width")] = body.getLargestRing().getRingWidthAsString()
                tmp[QCoreApplication.translate("additional", "ring_outer_rad")] = body.getLargestRing().getOuterRadiusAsString()

            tmp["meta_system_address"] = body.getSystemAddress()
            tmp["meta_body_id"] = body.getBodyId()

            data.append(tmp)

        return data

    def generateSunData(self, bodies, ringsOnly: bool):
        data = []
        for body in bodies:
            tmp = {
                QCoreApplication.translate("additional", "name"): body.getName(),
                QCoreApplication.translate("additional", "spectral_class"): body.getSpectralClass(),
                QCoreApplication.translate("additional", "mass"): body.getMassAsString(),
                QCoreApplication.translate("additional", "radius"): body.getRadiusAsString(),
                QCoreApplication.translate("additional", "temperature"): body.getTemperatureAsString(),
                QCoreApplication.translate("additional", "orbital_period"): body.getOrbitalPeriodAsString(),
                QCoreApplication.translate("additional", "rotation_period"): body. getRotationPeriodAsString(),
            }

            if ringsOnly:
                tmp[QCoreApplication.translate("additional", "ring_width")] = body.getLargestRing().getRingWidthAsString()
                tmp[QCoreApplication.translate("additional", "ring_outer_rad")] = body.getLargestRing().getOuterRadiusAsString()

            tmp["meta_system_address"] = body.getSystemAddress()
            tmp["meta_body_id"] = body.getBodyId()

            data.append(tmp)

        return data

    def getStarStatistics(self, type, criteria, sort, count, own, ringsOnly: bool=False):
        keyCriteria = self.getKeyFor(StatisticsData.STAR_CRITERIA, criteria)
        keySort = self.getKeyFor(StatisticsData.SORT_OPTIONS, sort)

        if type != '--all--':
            if "," in type:
                typeValue = type.split(",")
            else:
                typeValue = [type]

        whereParts = []
        whereValues = []

        query = """SELECT s.*, (r.outer_rad - r.inner_rad) AS ring_width, r.outer_rad as ring_outer_rad
                    FROM stars s
                    LEFT JOIN body_rings br ON br.star_id = s.id
                    LEFT JOIN rings r on br.ring_id = r.id
                """

        if type != '--all--':
            format_strings = ','.join(['%s'] * len(typeValue))
            whereParts.append("s.type IN (" + format_strings + ")")
            whereValues += tuple(typeValue)

        if own:
            whereParts.append("s.was_discovered = 0")

        if ringsOnly:
            whereParts.append("br.star_id IS NOT NULL")

        # append where
        if len(whereParts) > 0:
            query += " WHERE "
            query += " AND ".join(whereParts)

        # append order
        query += " ORDER BY " + keyCriteria + " " + keySort

        # append limit
        if count is not None:
            query += " LIMIT " + str(count)

        result = self.db.selectPlain(query, whereValues)
        stars = self.starService.getFromData(result)

        return self.generateSunData(stars, ringsOnly)
