from package.Datareader.SystemReader import SystemReader
from package.Datareader.PlanetReader import PlanetReader
from package.Datareader.StarReader import StarReader
from package.Datareader.FlightlogReader import FlightlogReader
from package.Service.StarService import StarService
from package.Service.PlanetService import PlanetService
from package.Helper.DistanceCalculator import DistanceCalculator
from package.Elite.System import System


class SearchService:
    def __init__(self):
        self.systemReader = SystemReader()
        self.planetReader = PlanetReader()
        self.starReader = StarReader()
        self.flightlogReader = FlightlogReader()
        self.starService = StarService()
        self.planetService = PlanetService()

    def getAutoSuggestResult(self, search: str) -> list:
        """
        Performs an auto-suggest search. First the systems will be searched, in case of no result
        bodies will be searched.
        :param search: string
        :return: list of found entries
        """
        result = self.systemReader.searchSystemByName(search)

        if len(result) == 0:
            result = self.planetReader.searchPlanetByName(search)

        if len(result) == 0:
            result = self.starReader.searchStarByName(search)

        return result

    def getSystem(self, systemAddress: int) -> System:
        """
        Creates the System object which contains all data about a system and will be used in the system view
        :param systemAddress: int
        :return: System
        """
        system = System()

        system = self.addBasicData(systemAddress, system)
        system = self.addAdditionalData(system)
        system = self.getCombinedBodies(systemAddress, system)
        system.processSystem()

        return system

    def addBasicData(self, systemAddress: int, system: System) -> System:
        """
        Add the basic system data from the db to the System object
        :param systemAddress: int
        :param system: System
        :return: System
        """
        systemData = self.systemReader.getSystemData(systemAddress)

        system.setName(systemData['name'])
        system.setPosX(systemData['pos_x'])
        system.setPosY(systemData['pos_y'])
        system.setPosZ(systemData['pos_z'])
        system.setScanComplete(systemData['scan_complete'])
        system.setSystemAddress(systemData['system_address'])

        return system

    def addAdditionalData(self, system: System) -> System:
        """
        Adds aditional data to the System object which will come from other sources
        :param system: System
        :return: System
        """
        # get current position
        currentPos = self.flightlogReader.getCurrentPosition()
        system.setPlayerPosition(currentPos)

        # distance
        systemPos = [system.getPosX(), system.getPosZ(), system.getPosY()]
        system.setDistance(DistanceCalculator.calculateDistance(currentPos, systemPos))

        # add stellar phenomena without a body_id
        result = self.systemReader.getStellarPhenomena(system.getSystemAddress())
        for entry in result:
            system.addStellarPhenomena(entry['name_localised'])

        return system

    def getCombinedBodies(self, systemAddress: int, system: System) -> System:
        """
        Add all bodies to the System object
        :param systemAddress: int
        :param system: System
        :return: System
        """
        stars = self.starService.getStarsForSystem(systemAddress)
        planets = self.planetService.getPlanetsForSystem(systemAddress)

        for star in stars:
            bodyId = star.getBodyId()
            system.addBody(bodyId, star)

        for planet in planets:
            bodyId = planet.getBodyId()
            system.addBody(bodyId, planet)

        return system

    def getAllBodiesForSystem(self, systemAddress: int) -> list:
        """
        Returns a list with all planets and stars of a system. no order!
        :param systemAddress: int
        :return: list
        """
        stars = self.starService.getStarsForSystem(systemAddress)
        planets = self.planetService.getPlanetsForSystem(systemAddress)

        return stars + planets
