from package.Datareader.PlanetReader import PlanetReader
from package.Journal.Events.Planet import Planet
from package.Journal.Events.SSASignal import SSASignal
from package.Journal.Events.CodexEntry import CodexEntry
from package.Journal.Events.ScanOrganic import ScanOrganic
from package.Config.ConfigManager import ConfigManager


class PlanetService:
    def __init__(self):
        self.reader = PlanetReader()

    def getPlanetsForSystem(self, systemAddress):
        planets = []

        planetsData = self.reader.getPlanetsForSystem(systemAddress)
        for entry in planetsData:
            signals = self.reader.getSignalsForPlanet(entry['id'])
            rings = self.reader.getRingsForPlanet(entry['id'])
            materials = self.reader.getMaterialsForPlanet(entry['id'])
            atmosphere = self.reader.getAtmosphereForPlanet(entry['id'])

            planet = Planet()
            planet.initFromDatabase(entry)
            planet.initSignalsFromDb(signals)
            planet.initRingsFromDatabase(rings)
            planet.initMaterialsFromDatabase(materials)
            planet.initAtmosphereCompositionFromDatabase(atmosphere)
            planet.setTemperatureFormat(ConfigManager.getConfigValue("temperature_format"))

            self.appendAdditionalPlanetData(planet)
            planets.append(planet)

        return planets

    def getPlanetsWorthScanning(self, systemAddress):
        planets = []

        planetsData = self.reader.getPlanetsWorthScanning(systemAddress)
        for entry in planetsData:
            planet = Planet()
            planet.initFromDatabase(entry)
            planet.setTemperatureFormat(ConfigManager.getConfigValue("temperature_format"))
            planets.append(planet)

        return planets

    def getPlanetByName(self, name):
        planetData = self.reader.getPlanetByName(name)
        planet = Planet()
        planet.initFromDatabase(planetData[0])
        planet.setTemperatureFormat(ConfigManager.getConfigValue("temperature_format"))

        return planet

    def getFromData(self, planetData):
        planets = []

        for entry in planetData:
            rings = self.reader.getRingsForPlanet(entry['id'])

            planet = Planet()
            planet.initFromDatabase(entry)
            planet.initRingsFromDatabase(rings)
            planet.setTemperatureFormat(ConfigManager.getConfigValue("temperature_format"))
            planets.append(planet)

        return planets

    def appendAdditionalPlanetData(self, planet: Planet):
        self.appendSSASignals(planet)
        self.appendCodexEntries(planet)
        self.appendOrganicSamples(planet)

    def appendSSASignals(self, planet: Planet):
        ssaRings = self.reader.getSsaSignalsForRing(planet.getInternalId())
        rings = planet.getRings()

        if len(ssaRings) > 0:
            for entry in ssaRings:
                ssa = SSASignal()
                ssa.initFromDatabase(entry)

                for ring in rings:
                    if ring.getInternalId() == entry['ring_id']:
                        ring.addSsaSignal(ssa)

    def appendCodexEntries(self, planet: Planet) -> None:
        codex = self.reader.getCodexEntriesForBody(planet.getInternalId())
        if len(codex) > 0:
            unique = []
            for entry in codex:
                ce = CodexEntry()
                ce.initFromDatabase(entry)

                if ce.getName() not in unique:
                    planet.addCodexEntry(ce)

                unique.append(ce.getName())

    def appendOrganicSamples(self, planet: Planet) -> None:
        scans = self.reader.getOrganicScansForBody(planet.getInternalId())
        if len(scans) > 0:
            for entry in scans:
                scan = ScanOrganic()
                scan.initFromDatabase(entry)
                planet.addOrganicScan(scan)
