from PyQt5 import QtCore, QtWidgets
from PyQt5.QtCore import QCoreApplication
from package.Observer.ObserverMessage import ObserverMessage
from package.Ui.Models.TableWorthScanning import TableWorthScanning
from package.Config.ConfigManager import ConfigManager
from package.Helper.DistanceCalculator import DistanceCalculator
from package.Notification.NotificationHandler import NotificationHandler
import locale


class SignalHandler(QtCore.QObject):
    def __init__(self):
        super(SignalHandler, self).__init__()
        self.view = None
        self.notifier = NotificationHandler()
        self.homeSystemName = ConfigManager.getConfigValue('home_system_name')
        self.setHomeSystem(
            ConfigManager.getConfigValue('home_system_pos_x'),
            ConfigManager.getConfigValue('home_system_pos_z'),
            ConfigManager.getConfigValue('home_system_pos_y')
        )

    def setView(self, view):
        self.view = view

    def setHomeSystem(self, x, z, y):
        self.homeSystemPos = [x, z, y]

    def parseSignal(self, observerData):
        if observerData.destination == ObserverMessage.OBSERVER_DESTINATION_POSITION:
            self.updatePosition(observerData.data)

        elif observerData.destination == ObserverMessage.OBSERVER_DESTINATION_STATUS:
            self.displayStatusMessage(observerData.data['message'])

        elif observerData.destination == ObserverMessage.OBSERVER_DESTINATION_ERROR:
            self.displayErrorMessage(observerData.data)

        elif observerData.destination == ObserverMessage.OBSERVER_DESTINATION_CHECKLOG:
            self.displayCheckMessage(observerData.data)

        elif observerData.destination == ObserverMessage.OBSERVER_DESTINATION_MAINSTAR:
            self.displayMainStarInfo(observerData.data)

        elif observerData.destination == ObserverMessage.OBSERVER_DESTINATION_WORTHSCANNING:
            self.displayWorthScanning(observerData.data)

        elif observerData.destination == ObserverMessage.OBSERVER_DESTINATION_CALCULATION:
            self.displayCalculationData(observerData.data)

        elif observerData.destination == ObserverMessage.OBSERVER_DESTINATION_CONTINUED:
            self.displayStatusMessage(QCoreApplication.translate("LineHandler", "continued_event"))

        elif observerData.destination == ObserverMessage.OBSERVER_DESTINATION_NOTIFY_WORTHSCANNING:
            self.executeNotifier(observerData.data['message'])

        elif observerData.destination == ObserverMessage.OBSERVER_DESTINATION_BIO_CALCULATION:
            self.displayBioCalculationData(observerData.data)

    def updatePosition(self, data):
        self.view.lineEditDiscoverySystem.setText(data['system'])
        self.view.lineEditDiscoveryPosX.setText(locale.format_string('%.2f', data['posX'], True))
        self.view.lineEditDiscoveryPosZ.setText(locale.format_string('%.2f', data['posZ'], True))
        self.view.lineEditDiscoveryPosY.setText(locale.format_string('%.2f', data['posY'], True))

        self.view.lineEditDiscoveryStarType.setText("")
        self.view.lineEditDiscoverySubClass.setText("")
        self.view.lineEditDiscoveryMass.setText("")
        self.view.lineEditDiscoveryRadius.setText("")
        self.view.lineEditDiscoveryTemperature.setText("")
        self.view.lineEditDiscoveryRotationPeriod.setText("")

        if "starType" in data and data['starType'] is not None:
            self.view.lineEditDiscoveryStarType.setText(data['starType'])

        distance = DistanceCalculator.calculateDistance(self.homeSystemPos, data['pos'])
        distance = locale.format_string('%.2f', distance, True)
        self.view.labelDiscoveryDistanceTo.setText(QCoreApplication.translate("MainWindow", "Distance to:").format(self.homeSystemName))
        self.view.labelDiscoveryLY.setText(QCoreApplication.translate("MainWindow", "LY").format(distance))

    def displayMessage(self, message):
        self.view.textBrowserDiscovery.append(message)

    def displayStatusMessage(self, message):
        message = "<font color='#000000'>" + message + "</font>"
        self.displayMessage(message)

    def displayErrorMessage(self, data):
        message = QCoreApplication.translate("LineHandler", "error") + ": " + data['error']
        formatedMessage = "<font color='#ff0000'><b>" + message + "</b></font>!!!"
        self.displayMessage(formatedMessage)

    def displayCheckMessage(self, data):
        if data.found is True:
            formatedMessage = "<font color='#2211AA'>"
            formatedMessage += QtCore.QCoreApplication.translate("Events", "found_poi")
            formatedMessage += "</font> <font color='#22AA22'><b>" + data.body + "</b></font><br>"
            formatedMessage += "<font color='#ff0000'><b>" + data.message + "</b></font><br>"
            self.displayMessage(formatedMessage)
            self.executeNotifier(data.message)

    def displayMainStarInfo(self, data):
        self.view.lineEditDiscoveryStarType.setText(QCoreApplication.translate("BodyTypes", data['starType']))
        self.view.lineEditDiscoverySubClass.setText(data['spectralType'] or '')
        self.view.lineEditDiscoveryMass.setText(data['mass'] or '')
        self.view.lineEditDiscoveryRadius.setText(data['radius'] or '')
        self.view.lineEditDiscoveryTemperature.setText(data['temperature'] or '')
        self.view.lineEditDiscoveryRotationPeriod.setText(data['rotationPeriod'] or '')

    def displayWorthScanning(self, data):
        # remove keys from data
        modelData = []
        for entry in data:
            modelData.append(data[entry])

        # set model
        self.view.worthScanningModel = TableWorthScanning(modelData)
        self.view.worthScanningSortProxy.setSourceModel(self.view.worthScanningModel)
        self.view.tableViewDiscoveryWorthScanning.setModel(self.view.worthScanningSortProxy)

        # set column sizes
        header = self.view.tableViewDiscoveryWorthScanning.horizontalHeader()
        for x in range(3):
            if x == 1:
                header.setSectionResizeMode(x, QtWidgets.QHeaderView.Stretch)
            else:
                header.setSectionResizeMode(x, QtWidgets.QHeaderView.ResizeToContents)

        self.view.tableViewDiscoveryWorthScanning.sortByColumn(0, QtCore.Qt.AscendingOrder)

    def displayCalculationData(self, data):
        sessionValue = locale.format_string('%.0f', data['sessionValue'], True)
        totalValue = locale.format_string('%.0f', data['totalValue'], True)

        self.view.lineEditDiscoverySessionValue.setText(sessionValue)
        self.view.lineEditDiscoveryTotalValue.setText(totalValue)

    def executeNotifier(self, text):
        self.notifier.notify(text)

    def displayBioCalculationData(self, data):
        sessionValue = locale.format_string('%.0f', data['sessionValue'], True)
        totalValue = locale.format_string('%.0f', data['totalValue'], True)

        self.view.lineEditDiscoveryBioSession.setText(sessionValue)
        self.view.lineEditDiscoveryBioTotalValue.setText(totalValue)
