import json
import sys
import traceback
import logging
from time import sleep
from os import SEEK_SET
from os.path import basename
from PyQt5 import QtCore
from PyQt5.QtCore import QCoreApplication
from watchdog.events import FileSystemEventHandler
from watchdog.observers import Observer
from package.Observer.ObserverMessage import ObserverMessage
from package.Config.ConfigManager import ConfigManager
from package.Base.LineHandler import LineHandler
from package.Helper.FileHelper import FileHelper
from package.Elite.Events import EVENTS
from package.Helper.ClassLoader import ClassLoader

log = logging.getLogger(__name__)


class FileWatcher(QtCore.QObject, FileSystemEventHandler):
    fileWatcherSignal = QtCore.pyqtSignal(object)
    directDisplaySignal = QtCore.pyqtSignal(str)
    SLEEPTIME = 0.5

    def __init__(self):
        super().__init__()
        self.journalDir = ConfigManager.getConfigValue("journal_directory")
        self.newestLogFile = None
        self.logOpen = False
        self.observer = None
        self.running = True
        self.scanner = None
        self.fileHelper = FileHelper()

    def start(self):
        latestLog = self.fileHelper.getLatestLogFile(self.journalDir)

        msg = QCoreApplication.translate("Filewatcher", "observing_dir") + "<br>" + self.journalDir
        self.directDisplaySignal.emit(msg)

        if latestLog is not None:
            self.directDisplaySignal.emit(
                QCoreApplication.translate("Filewatcher", "last_logfile").format(latestLog)
            )

            logFile = self.journalDir + "/" + latestLog
            if self.isGameRunning(logFile):
                self.directDisplaySignal.emit(QCoreApplication.translate("Filewatcher", "game_running"))
                self.newestLogFile = logFile
            else:
                self.directDisplaySignal.emit(QCoreApplication.translate("Filewatcher", "game_not_running"))

        # start observer
        self.observer = Observer()
        self.observer.schedule(self, self.journalDir)
        self.observer.start()

    def stop(self):
        if self.observer is not None:
            self.observer.stop()
            self.observer.join()

        self.newestLogFile = None
        self.running = False

    # if the latest logfile contains the Shutdown event, the game is not running
    # if it doesn't we assume that the game is running - this might be not true for all situations
    # but it's ok for now
    def isGameRunning(self, logFile):
        isRunning = True
        filePointer = open(logFile, 'rb', 0)

        for stringline in filePointer:
            if len(stringline) == 0:
                break

            if b'"event":"Shutdown"' in stringline:
                isRunning = False

        filePointer.close()

        return isRunning

    # watchdog event for newly created files
    def on_created(self, event):
        if not event.is_directory and self.fileHelper.isLogFile(basename(event.src_path)):
            self.directDisplaySignal.emit(
                QCoreApplication.translate("Filewatcher", "new_journal_detected").format(event.src_path)
            )
            self.newestLogFile = event.src_path

    def watcher(self):
        currentLogFile = self.newestLogFile
        filePointer = None
        filePosition = 0

        # if a logfile exists, open it
        if currentLogFile is not None:
            self.directDisplaySignal.emit(
                QCoreApplication.translate("Filewatcher", "trying_to_open_journal").format(currentLogFile)
            )
            filePointer = open(currentLogFile, 'rb', 0)

        while self.running:
            # has the log file changed?
            if currentLogFile != self.newestLogFile:
                self.directDisplaySignal.emit(QCoreApplication.translate("Filewatcher", "switching_journal"))
                # close current file
                if filePointer:
                    self.directDisplaySignal.emit(QCoreApplication.translate("Filewatcher", "closing_journal"))
                    filePointer.close()

                # set new file as current file
                currentLogFile = self.newestLogFile

                if currentLogFile:
                    self.directDisplaySignal.emit(
                        QCoreApplication.translate("Filewatcher", "open_new_journal").format(currentLogFile)
                    )
                    filePointer = open(currentLogFile, 'rb', 0)
                    filePosition = 0

            if currentLogFile:
                # set file pointer to last position
                filePointer.seek(filePosition, SEEK_SET)

                # read all new lines
                for stringline in filePointer:
                    if len(stringline) == 0:
                        break

                    try:
                        logEntry = json.loads(stringline)
                        self.lineHandler.processLine(logEntry)
                    except:
                        message = str(sys.exc_info()[0])
                        message += traceback.format_exc()

                        # include line from file if it is a string (a json line should be one)
                        if isinstance(stringline, str):
                            message += str(stringline)

                        log.critical(message)
                        print(message)
                        self.emitMessage(ObserverMessage.createErrorMessage(message))

                # save last file position
                filePosition = filePointer.tell()

            sleep(self.SLEEPTIME)

    def setupLineHandler(self):
        packages = [
            "package.Journal.Handler.Observer.",
            "package.Journal.Handler.Import.",
        ]

        self.lineHandler = LineHandler()

        for packageNameBase in packages:
            for eventName in EVENTS:
                moduleName = eventName + "Event"
                packageName = packageNameBase + moduleName

                eventInstance = ClassLoader.getClass(packageName, moduleName)

                if eventInstance is not None:
                    eventInstance.eventSignal.connect(self.emitMessage)

                    # set worth scanning scanner for events
                    tmpOp = getattr(eventInstance, "setScanner", None)
                    if callable(tmpOp):
                        eventInstance.setScanner(self.scanner)
                    self.lineHandler.registerSubscriber(eventInstance)

    def emitMessage(self, message):
        self.fileWatcherSignal.emit(
            message
        )

    def setScanner(self, scanner):
        self.scanner = scanner

    def getScanner(self):
        return self.scanner
