from abc import ABC, abstractmethod
from PyQt5.QtCore import QCoreApplication
from package.Observer.CheckResult import CheckResult
from package.Database.DbFactory import DbFactory


class CheckerAbstract(ABC):
    CHECKER_TYPE_BODY = "BODY"
    CHECKER_TYPE_SYSTEM = "SYSTEM"
    CHECKER_TYPE = CHECKER_TYPE_BODY

    def __init__(self):
        self.result = CheckResult()
        self.name = self.__class__.__name__
        self.db = DbFactory.getDatabase()
        self.config = {}
        self.initConfigValues()

        super().__init__()

    @abstractmethod
    def check(self, body=None):
        self.result = CheckResult()

    def translate(self, text, context="Checker"):
        return QCoreApplication.translate(context, text)

    def getName(self) -> str:
        """
        Return the name of the checker - is the class name currently
        :return: string
        """
        return self.name

    def getConfigValue(self, name: str) -> str:
        """
        Return a config value for the checker
        :param name: string
        :return: string
        """
        if name in self.config:
            return self.config[name]
        else:
            return None

    def initConfigValues(self) -> None:
        """
        Read all config values for the checker. Called when class is instanciated
        :return: None
        """
        data = self.db.select("checker_configuration", ["config_name", "config_value"], {'checker_name': self.name})
        for entry in data:
            self.config[entry["config_name"]] = entry["config_value"]

    def hasConfigValues(self) -> bool:
        """
        Determines if the checker has config values. used in the frontend to display a button to open the
        checker settings. Each checker class will overwrite this value if necessary.
        :return: bool
        """
        return False

    def getConfigFields(self) -> list:
        """
        Returns the available configuration fields for the checker. used in the UI to display the form fields.
        :return: dict: {'type': 'float|dropdown|int' 'name': 'name_of_config_value', 'value': 'value_of_config_value'}
        """
        return []
