from package.Observer.CheckerAbstract import CheckerAbstract
from package.Database.DbFactory import DbFactory
from package.Elite.StarTypes import StarTypes


class CheckStarRecords(CheckerAbstract):
    starCriteria = [
        "Radius",
        "StellarMass",
        "SurfaceTemperature",
        "DistanceFromArrivalLS",
        "RotationPeriod",
    ]

    journalToDb = {
        "Radius": "radius",
        "StellarMass": "mass",
        "SurfaceTemperature": "temperature",
        "DistanceFromArrivalLS": "distance",
        "RotationPeriod": "rotation_period",
    }

    records = {}

    def __init__(self):
        super(CheckStarRecords, self).__init__()
        self.initRecords()

    def check(self, body):
        super().check(body)

        starType = body.getType()
        messages = []

        if body.wasDiscovered():
            return self.result

        if starType is not None:
            for criteria in self.records:
                if starType in self.records[criteria]:
                    testValueMax = self.records[criteria][starType]["max"]
                    testValueMin = self.records[criteria][starType]["min"]
                    bodyValue = self.getBodyRawValue(body, criteria)

                    if (bodyValue is not None) and (bodyValue > testValueMax):
                        trStarType = self.translate(starType, "BodyTypes")

                        messages.append(self.translate("checker_record_max").format(
                            self.translate(criteria),
                            trStarType,
                            self.getBodyValue(body, criteria)
                        ))

                        # set as new record
                        self.records[criteria][starType]["max"] = bodyValue


                    if (bodyValue is not None) and (bodyValue < testValueMin):
                        trStarType = self.translate(starType, "BodyTypes")

                        messages.append(self.translate("checker_record_min").format(
                            self.translate(criteria),
                            trStarType,
                            self.getBodyValue(body, criteria)
                        ))

                        # set as new record
                        self.records[criteria][starType]["min"] = bodyValue

        if len(messages) > 0:
            self.result.found = True
            self.result.body = body.getName()
            self.result.message = '<br>'.join(messages)

        return self.result

    def initRecords(self):
        db = DbFactory.getDatabase()

        selects = []
        for criteria in self.starCriteria:
            dbField = self.journalToDb[criteria]
            selects.append("MAX(" + dbField + ") as max" + dbField)
            selects.append("MIN(" + dbField + ") as min" + dbField)

        for key in StarTypes.TYPES:
            starType = StarTypes.TYPES[key]
            data = {"type": starType, "was_discovered": 0}
            result = db.select("stars", selects, data)

            for criteria in self.starCriteria:
                dbField = self.journalToDb[criteria]

                if criteria not in self.records:
                    self.records[criteria] = {}

                if starType not in self.records[criteria]:
                    self.records[criteria][starType] = {
                        "max": 0,
                        "min": 0
                    }

                if result[0]["max" + dbField] is not None:
                    self.records[criteria][starType]["max"] = result[0]["max" + dbField]

                if result[0]["min" + dbField] is not None:
                    self.records[criteria][starType]["min"] = result[0]["min" + dbField]

    def getBodyRawValue(self, body, criteria):
        if criteria == "Radius":
            return body.getRadius()
        elif criteria == "StellarMass":
            return body.getMass()
        elif criteria == "SurfaceTemperature":
            return body.getTemperature()
        elif criteria == "DistanceFromArrivalLS":
            return body.getDistance()
        elif criteria == "RotationPeriod":
            return body.getRotationPeriod()
        else:
            return None

    def getBodyValue(self, body, criteria):
        if criteria == "Radius":
            return body.getRadiusAsString(6)
        elif criteria == "StellarMass":
            return body.getMassAsString(6)
        elif criteria == "SurfaceTemperature":
            return body.getTemperatureAsString(6)
        elif criteria == "DistanceFromArrivalLS":
            return body.getDistanceAsString()
        elif criteria == "RotationPeriod":
            return body.getRotationPeriodAsString()
        else:
            return None
