from package.Observer.CheckerAbstract import CheckerAbstract
from package.Database.DbFactory import DbFactory
from package.Elite.PlanetTypes import PlanetTypes
from package.Journal.Events.Planet import Planet


class CheckPlanetRecords(CheckerAbstract):
    planetCriteria = [
        "Radius",
        "MassEM",
        "SurfaceGravity",
        "SurfaceTemperature",
        "Eccentricity",
        "DistanceFromArrivalLS"
    ]

    journalToDb = {
        "Radius": "radius",
        "MassEM": "mass",
        "SurfaceTemperature": "temperature",
        "SurfaceGravity": "gravity",
        "Eccentricity": "eccentricity",
        "DistanceFromArrivalLS": "distance"
    }

    records = {}

    def __init__(self):
        super(CheckPlanetRecords, self).__init__()
        self.initRecords()

    def check(self, body):
        super().check(body)

        if not isinstance(body, Planet):
            return self.result

        planetClass = body.getType()
        messages = []

        if body.wasDiscovered() or body.wasMapped():
            return self.result

        if planetClass is not None:
            for criteria in self.records:
                if planetClass in self.records[criteria]:
                    testValueMax = self.records[criteria][planetClass]["max"]
                    testValueMin = self.records[criteria][planetClass]["min"]
                    bodyValue = self.getBodyRawValue(body, criteria)

                    if (bodyValue is not None) and (bodyValue > testValueMax):
                        trPlanetType = self.translate(planetClass, "BodyTypes")

                        messages.append(self.translate("checker_record_max").format(
                            self.translate(criteria),
                            trPlanetType,
                            self.getBodyValue(body, criteria)
                        ))

                        # set as new record
                        self.records[criteria][planetClass]["max"] = bodyValue


                    if (bodyValue is not None) and (bodyValue < testValueMin):
                        trPlanetType = self.translate(planetClass, "BodyTypes")

                        messages.append(self.translate("checker_record_min").format(
                            self.translate(criteria),
                            trPlanetType,
                            self.getBodyValue(body, criteria)
                        ))

                        # set as new record
                        self.records[criteria][planetClass]["min"] = bodyValue

        if len(messages) > 0:
            self.result.found = True
            self.result.body = body.getName()
            self.result.message = '<br>'.join(messages)

        return self.result

    def initRecords(self):
        db = DbFactory.getDatabase()

        selects = []
        for criteria in self.planetCriteria:
            dbField = self.journalToDb[criteria]
            selects.append("MAX(" + dbField + ") as max" + dbField)
            selects.append("MIN(" + dbField + ") as min" + dbField)

        for key in PlanetTypes.TYPES:
            planetType = PlanetTypes.TYPES[key]
            data = {"type": planetType, "was_discovered": 0}
            result = db.select("bodies", selects, data)

            for criteria in self.planetCriteria:
                dbField = self.journalToDb[criteria]

                if criteria not in self.records:
                    self.records[criteria] = {}

                if planetType not in self.records[criteria]:
                    self.records[criteria][planetType] = {
                        "max": 0,
                        "min": 0
                    }

                if result[0]["max" + dbField] is not None:
                    self.records[criteria][planetType]["max"] = result[0]["max" + dbField]

                if result[0]["min" + dbField] is not None:
                    self.records[criteria][planetType]["min"] = result[0]["min" + dbField]

    def getBodyRawValue(self, body, criteria):
        if criteria == "Radius":
            return body.getRadius()
        elif criteria == "MassEM":
            return body.getMass()
        elif criteria == "SurfaceGravity":
            return body.getGravity()
        elif criteria == "SurfaceTemperature":
            return body.getTemperature()
        elif criteria == "Eccentricity":
            return body.getEccentricity()
        elif criteria == "DistanceFromArrivalLS":
            return body.getDistance()
        else:
            return None

    def getBodyValue(self, body, criteria):
        if criteria == "Radius":
            return body.getRadiusAsString(6)
        elif criteria == "MassEM":
            return body.getMassAsString(6)
        elif criteria == "SurfaceGravity":
            return body.getGravityAsString(6)
        elif criteria == "SurfaceTemperature":
            return body.getTemperatureAsString(6)
        elif criteria == "Eccentricity":
            return body.getEccentricityAsString(6)
        elif criteria == "DistanceFromArrivalLS":
            return body.getDistanceAsString()
        else:
            return None
