import locale
from package.Observer.CheckerSystemAbstract import CheckerSystemAbstract
from package.Journal.Events.Planet import Planet


class CheckMoonCloseToRing(CheckerSystemAbstract):
    CONFIG_NAME_TIMES_RADIUS = "times_radius"

    def check(self, systemState):
        super().check(systemState)

        criteriaTimesRadius = int(self.getConfigValue(self.CONFIG_NAME_TIMES_RADIUS))

        system = self.systemState.getSystem()
        for i in system:
            body = system[i]
            parentId = body.getParentId()

            if parentId is not None and parentId in system:
                parentObject = system[parentId]

                rings = parentObject.getRings()
                bodyAxis = body.getSemiMajorAxis()
                radius = body.getRadius()
                inclination = body.getInclination()

                if (bodyAxis is None) or (radius is None) or (len(rings) == 0):
                    continue

                for ring in rings:
                    ringName = ring.getName()
                    ringOuterRad = ring.getOuterRadius()
                    ringInnerRad = ring.getInnerRadius()

                    if (ringOuterRad is None) or (ringInnerRad is None):
                        continue

                    if "Belt" not in ringName:
                        n1 = abs(bodyAxis - ringOuterRad)
                        n2 = abs(ringInnerRad - bodyAxis)

                        if n1 < n2:
                            seperation = n1
                        else:
                            seperation = n2

                        if seperation < (radius * criteriaTimesRadius):
                            inclinationTxt = ""
                            landableTxt = ""

                            if inclination is not None:
                                if inclination < -20 or inclination > 20:
                                    inclinationTxt = self.translate("high_inclination")

                            # is moon landable
                            if isinstance(body, Planet) and body.isLandable():
                                landableTxt = "(" + self.translate("landable") + ")"

                            seperation = seperation / 1000
                            message = self.translate("checker_moon_close_to_ring").format(
                                locale.format_string('%.2f', seperation, True),
                                landableTxt,
                                inclinationTxt
                            )

                            self.addMessage(
                                message,
                                body.getName(),
                            )

                            break

        return self.results

    def hasConfigValues(self) -> bool:
        return True

    def getConfigFields(self) -> list:
        """
        Returns the available configuration fields for the checker. used in the UI to display the form fields.
        :return: dict
        """
        return [
            {
                'type': 'int',
                'name': self.CONFIG_NAME_TIMES_RADIUS,
                'value': self.getConfigValue(self.CONFIG_NAME_TIMES_RADIUS)
            }
        ]
