from package.Calculator.Calculator import Calculator
from package.Journal.Events.Planet import Planet
from package.Journal.Events.Star import Star
from package.Calculator.OrganicsCalculator import OrganicsCalculator


class CalculationHandler:
    _instance = None

    def __init__(self, db):
        self.db = db
        self.sessionValue = 0
        self.bioSessionValue = 0
        self.totalValue = self.initTotalValue()
        self.bioTotalValue = self.initBioTotalValue()

        CalculationHandler._instance = self

    @staticmethod
    def getInstance(db):
        """ Fake singleton for now """
        if CalculationHandler._instance == None:
            CalculationHandler(db)
        return CalculationHandler._instance

    def handleSimpleCalculation(self, body):
        bodyName = body.getName()
        value = Calculator.calculate(body, False, False)

        # if object exists in db it was calculated before
        if self.objectExists(body):
            return

        if value > 0 and not self.calculationExists(bodyName):
            self.insertCalculation(bodyName, value)
            self.addValue(value)

    def handleDetailCalculation(self, entity):
        bodyName = entity.getBodyName()
        probesUsed = entity.getProbesUsed()
        effiencyTarget = entity.getEfficiencyTarget()
        effiencyBonus = probesUsed <= effiencyTarget

        # Rings have no value and are stored in a different table. So don't perform a detail calculation for rings
        if bodyName.endswith(' Ring'):
            print("Rings don't need to be calculated: " + bodyName)
            return

        bodyValues = self.getBodyData(bodyName)
        if bodyValues is None:
            print("No DB data for " + bodyName + " - skipping detail calculation")
            return

        # if object was already detail scanned, do not scan it again
        if bodyValues['detail_scan'] == 1:
            print(bodyName + " was already detail scanned. Skipping calculation")
            return

        value = Calculator.calculateFromData(
            bodyValues['type'],
            bodyValues['mass'],
            bodyValues['terraformable'],
            True,
            bodyValues['was_discovered'],
            bodyValues['was_mapped'],
            effiencyBonus
        )

        if not self.calculationExists(bodyName):
            self.insertCalculation(bodyName, value)
            self.addValue(value)
            print("Adding in update - this should not happen: " + str(value))
        else:
            correctionValue = self.getCorrectionValue(bodyName)
            self.updateCalculation(bodyName, value)
            self.correctValue(value, correctionValue)

    def getBodyData(self, bodyName):
        result = self.db.select('bodies', ['type', 'mass', 'terraformable', 'was_discovered', 'was_mapped', 'detail_scan'], {'name': bodyName}, None, 1)
        if result == []:
            return None
        else:
            return result[0]

    def getCorrectionValue(self, bodyName):
        result = self.db.select('calculation', ['value'], {'name': bodyName}, None, 1)
        if result == []:
            return 0
        else:
            return result[0]['value']

    def calculationExists(self, bodyName):
        result = self.db.select('calculation', ['id'], {'name': bodyName}, None, 1)
        if result == []:
            exists = False
        else:
            exists = True

        return exists

    def insertCalculation(self, bodyName, value):
        self.db.insert('calculation', {'name': bodyName, 'value': value})

    def updateCalculation(self, bodyName, value):
        self.db.update('calculation', {'value': value}, {'name': bodyName})

    def initTotalValue(self):
        result = self.db.select('calculation', ['SUM(value) as total'], {})

        if result[0]['total'] is None:
            return 0
        else:
            return result[0]['total']

    def getTotalValue(self):
        return self.totalValue

    def getSessionValue(self):
        return self.sessionValue

    def getBioTotalValue(self):
        return self.bioTotalValue

    def getBioSessionValue(self):
        return self.bioSessionValue

    def addValue(self, value):
        self.sessionValue += value
        self.totalValue += value

    def addBioValue(self, value):
        self.bioSessionValue += value
        self.bioTotalValue += value

    def correctValue(self, value, correction):
        self.sessionValue -= correction
        self.sessionValue += value
        self.totalValue -= correction
        self.totalValue += value

    def objectExists(self, body):
        bodyName = body.getName()

        objectExists = False
        if isinstance(body, Star):
            objectExists = self.starExists(bodyName)
        elif isinstance(body, Planet):
            objectExists = self.planetExists(bodyName)

        return objectExists

    def starExists(self, bodyName):
        result = self.db.select('stars', ['id'], {'name': bodyName}, None, 1)
        if result == []:
            return False
        else:
            return True

    def planetExists(self, bodyName):
        result = self.db.select('bodies', ['id'], {'name': bodyName}, None, 1)
        if result == []:
            return False
        else:
            return True

    def resetValues(self) -> None:
        """
        Reset all calculation values (not in DB, only in view)
        :return: None
        """
        self.resetBodyScanValues()
        self.resetBioScanValues()

    def resetBodyScanValues(self) -> None:
        """
        Reset body scan values (not in DB, only in view)
        :return: None
        """
        self.sessionValue = 0
        self.totalValue = 0

    def resetBioScanValues(self) -> None:
        """
        Reset bio calculation values (not in DB, only in view)
        :return: None
        """
        self.bioSessionValue = 0
        self.bioTotalValue = 0

    def initBioTotalValue(self):
        result = self.db.select('bio_calculation', ['SUM(value) as total'], {})

        if result[0]['total'] is None:
            return 0
        else:
            return result[0]['total']

    def insertBioCalculation(self, species, value):
        self.db.insert('bio_calculation', {'name': species, 'value': value})

    def handleBioCalculation(self, species):
        credits = OrganicsCalculator.getCreditsForOrganics(species.lower())
        self.insertBioCalculation(species, credits)
        self.addBioValue(credits)
