from PyQt5.QtCore import QCoreApplication
from .ObserverAbstract import ObserverAbstract
from package.Database.DbFactory import DbFactory
from package.Observer.ObserverMessage import ObserverMessage
from package.Observer.CalculationHandler import CalculationHandler
from package.Observer.SessionStatisticHandler import SessionStatisticHandler
from package.Journal.Events.Star import Star
from package.Journal.Events.Planet import Planet


class ScanEvent(ObserverAbstract):
    listenToEvent = "Scan"

    def notify(self, body):
        # Ignore Belt Clusters and stuff for now
        if body is None:
            return

        # inject body signals in planet?
        if isinstance(body, Planet):
            # check if body signals are available
            bodySignalEvent = self.system.getBodySignalEvent(body.getBodySignalKey())
            if bodySignalEvent is not None:
                body.addSignalsFromJournal(bodySignalEvent.getSignals())

        if not self.system.isSystemComplete():
            self.system.addBody(body)

            results = self.analyzer.analyzeBody(body)

            # emit messages from analizer. should be done in analyzer in the future
            if len(results) > 0:
                for message in results:
                    self.eventSignal.emit(message)

            self.doCalculation(body)

        # set main star info
        self.createMainStarMessage(body)

        # check if object is worth scanning
        self.checkWorthScanning(body)

        # add statistic data
        SessionStatisticHandler.getInstance().addScanData(body)

    def doCalculation(self, body):
        db = DbFactory.getDatabase()
        CalculationHandler.getInstance(db).handleSimpleCalculation(body)

        self.eventSignal.emit(
            ObserverMessage.createCalculationMessage(
                CalculationHandler.getInstance(db).getSessionValue(),
                CalculationHandler.getInstance(db).getTotalValue()
            )
        )

    def createMainStarMessage(self, body):
        if isinstance(body, Star) and body.isMainStar():
            message = ObserverMessage.createMainStarMessage(
                {
                    "spectralType": body.getSpectralClass(),
                    "radius": body.getRadiusAsString(),
                    "mass": body.getMassAsString(),
                    "starType": body.getType(),
                    "temperature": body.getTemperatureAsString(),
                    "rotationPeriod": body.getRotationPeriodAsString(),
                }
            )
            self.eventSignal.emit(message)

    def checkWorthScanning(self, body):
        scanResult = self.scanner.isWorthScanning(body)

        if scanResult.isSuccessful():
            isTerraformable = body.isTerraformable()

            number = self.getBodyNumber(
                body.getName(),
                body.getSystemName()
            )

            translated = QCoreApplication.translate("worth_scanning", scanResult.getName())

            if self.system.isSetObjectWorthScanning(number) is False:
                self.system.addWorthScanning(
                    number,
                    translated,
                    body.getDistanceAsString(0),
                    isTerraformable,
                    False
                )
                self.eventSignal.emit(ObserverMessage.createNotifyWorthScanningMessage(translated))

            self.eventSignal.emit(ObserverMessage.createWorthScanningMessage(self.system.getObjectsWorthScanning()))
