from PyQt5.QtCore import QCoreApplication
from .ObserverAbstract import ObserverAbstract
from package.Observer.ObserverMessage import ObserverMessage
from package.Service.StarService import StarService
from package.Service.SearchService import SearchService
from package.Datareader.FlightlogReader import FlightlogReader

class LocationEvent(ObserverAbstract):
    listenToEvent = "Location"

    def __init__(self):
        super().__init__()
        self.starService = StarService()
        self.searchService = SearchService()
        self.flightlogReader = FlightlogReader()

    def notify(self, event):
        self.system.setCurrentSystem(event.getStarSystem())

        # check last flightlog position
        lastFlightLogSystemAddress = self.flightlogReader.getLastSystemAddress()

        if event.getSystemAddress() != lastFlightLogSystemAddress:
            # probably a carrier jump or the game crashed before entering a new system and starts in the new system again
            # in this case clear the system
            self.system.resetSystem()
            self.system.clearObjectsWorthScanning()

        # update position
        self.eventSignal.emit(
            ObserverMessage.createPositionMessage(
                {
                    "system": event.getStarSystem(),
                    "posX": event.getPosX(),
                    "posY": event.getPosY(),
                    "posZ": event.getPosZ(),
                    "pos": event.getPos()
                }
            )
        )

        # update main star and worth scanning from db
        self.setMainStarInfo(event.getSystemAddress())
        self.setWorthScanningInfo(event.getSystemAddress())

        # add current z position
        self.system.setPosZ(event.getPosZ())

    def setMainStarInfo(self, systemAddress):
        mainStar = self.starService.getMainStar(systemAddress)

        # update main star
        if mainStar:
            self.eventSignal.emit(
                ObserverMessage.createMainStarMessage(
                    {
                        "spectralType": mainStar.getSpectralClass(),
                        "radius": mainStar.getRadiusAsString(),
                        "mass": mainStar.getMassAsString(),
                        "starType": mainStar.getType(),
                        "temperature": mainStar.getTemperatureAsString(),
                        "rotationPeriod": mainStar.getRotationPeriodAsString()
                    }
                )
            )

    def setWorthScanningInfo(self, systemAddress):
        systemBodies = self.searchService.getAllBodiesForSystem(systemAddress)

        # update worth scanable objects
        for body in systemBodies:
            scanResult = self.scanner.isWorthScanning(body)

            if scanResult.isSuccessful():
                number = self.getBodyNumber(
                    body.getName(),
                    body.getSystemName()
                )

                self.system.addWorthScanning(
                    number,
                    QCoreApplication.translate("worth_scanning", scanResult.getName()),
                    body.getDistanceAsString(0),
                    body.isTerraformable(),
                    body.isDetailScaned()
                )

        self.eventSignal.emit(
            ObserverMessage.createWorthScanningMessage(self.system.getObjectsWorthScanning())
        )
