from PyQt5.QtCore import QCoreApplication
from package.Journal.Handler.Observer.ObserverAbstract import ObserverAbstract
from package.Observer.ObserverMessage import ObserverMessage
from package.Observer.SessionStatisticHandler import SessionStatisticHandler
from package.Service.StarService import StarService
from package.Service.SearchService import SearchService


class FSDJumpEvent(ObserverAbstract):
    listenToEvent = "FSDJump"

    def __init__(self):
        super().__init__()
        self.starService = StarService()
        self.searchService = SearchService()

    def notify(self, event):
        # if we have objects left, analyze them
        if self.system.getCount() > 0:
            self.eventSignal.emit(
                ObserverMessage.createStatusMessage(
                    QCoreApplication.translate("Events", "Analyzing remaining {0} bodies because we are leaving the system").format(
                        str(self.system.getCount()))
                )
            )

            # analyze system
            results = self.analyzer.analyzeSystem(self.system)

            # emit messages from analizer. should be done in analyzer in the future
            if len(results) > 0:
                for message in results:
                    self.eventSignal.emit(message)

        # get new star class (as a translated string)
        starTranslated = self.getNewStarClass(event)

        # send "entering new system" message
        self.sendNewSystemMessage(event, starTranslated)

        # send position message
        self.eventSignal.emit(
            ObserverMessage.createPositionMessage(
                {
                    "system": event.getStarSystem(),
                    "posX": event.getPosX(),
                    "posY": event.getPosY(),
                    "posZ": event.getPosZ(),
                    "starType": starTranslated,
                    "pos": event.getPos()
                }
            )
        )

        # clean system data
        self.resetSystem()

        # set current system
        self.system.setCurrentSystem(event.getStarSystem())

        # update main star & worth scanning from db
        self.setMainStarInfo(event.getSystemAddress())
        self.setWorthScanningInfo(event.getSystemAddress())

        # add to session statistic
        SessionStatisticHandler.getInstance().addJumpData(event)

        # add current z position
        self.system.setPosZ(event.getPosZ())

    def sendNewSystemMessage(self, event, starTranslated):
        if starTranslated != "":
            message = QCoreApplication.translate("Events", "entering_system_with_class").format(event.getStarSystem(), starTranslated)
        else:
            message = QCoreApplication.translate("Events", "entering_system").format(event.getStarSystem())

        self.eventSignal.emit(
            ObserverMessage.createStatusMessage(
                "<br>" + message
            )
        )

    def getNewStarClass(self, event):
        starTranslated = ""
        if self.system.getNextStarClass() is not None and self.system.getNextStarSystem() == event.getStarSystem():
            starTranslated = QCoreApplication.translate("BodyTypes", self.system.getNextStarClass())

        return starTranslated

    def resetSystem(self):
        self.system.reset()

        # send an empty worth scanning list, cause we are in a new system
        self.eventSignal.emit(ObserverMessage.createWorthScanningMessage(self.system.getObjectsWorthScanning()))

    def setMainStarInfo(self, systemAddress):
        mainStar = self.starService.getMainStar(systemAddress)

        if mainStar:
            self.eventSignal.emit(
                ObserverMessage.createMainStarMessage(
                    {
                        "spectralType": mainStar.getSpectralClass(),
                        "radius": mainStar.getRadiusAsString(),
                        "mass": mainStar.getMassAsString(),
                        "starType": mainStar.getType(),
                        "temperature": mainStar.getTemperatureAsString(),
                        "rotationPeriod": mainStar.getRotationPeriodAsString()
                    }
                )
            )

    def setWorthScanningInfo(self, systemAddress):
        systemBodies = self.searchService.getAllBodiesForSystem(systemAddress)

        # update worth scanable objects
        for body in systemBodies:
            scanResult = self.scanner.isWorthScanning(body)

            if scanResult.isSuccessful():
                number = self.getBodyNumber(
                    body.getName(),
                    body.getSystemName()
                )

                self.system.addWorthScanning(
                    number,
                    QCoreApplication.translate("worth_scanning", scanResult.getName()),
                    body.getDistanceAsString(0),
                    body.isTerraformable(),
                    body.isDetailScaned()
                )

        self.eventSignal.emit(
            ObserverMessage.createWorthScanningMessage(self.system.getObjectsWorthScanning())
        )
