import locale
from PyQt5.QtCore import QCoreApplication
from decimal import *
from package.Journal.Events.BodyEntity import BodyEntity
from package.Helper.Converter import Converter


class Star(BodyEntity):
    def __init__(self):
        super().__init__()

        self._mass = None
        self._isStar = True
        self._isMainStar = False
        self._subClass = None
        self._absoluteMagnitude = None
        self._age = None
        self._luminosity = None

    def initFromJournal(self, data):
        super().initFromJournal(data)

        self._mass = Decimal(self.getParam(data, 'StellarMass', 0))
        self._subClass = self.getParam(data, 'Subclass', 0)
        self._absoluteMagnitude = self.getParam(data, 'AbsoluteMagnitude', 0)
        self._age = self.getParam(data, 'Age_MY', 0)
        self._luminosity = self.getParam(data, 'Luminosity', 0)

        # main star - simply where distance = 0 ?
        self._isMainStar = False
        if self._distance == 0:
            self._isMainStar = True

    def initFromDatabase(self, data):
        super().initFromDatabase(data)

        if data is not None:
            self._mass = Decimal(data['mass'])
            self._subClass = data['subclass']
            self._absoluteMagnitude = Decimal(data['absolute_magnitude'])
            self._age = Decimal(data['age'])
            self._luminosity = data['luminosity']

            # main star - simply where distance = 0 ?
            self._isMainStar = False
            if data['is_main_star'] == 1:
                self._isMainStar = True

    def getSpectralClass(self):
        spectralClass = ""

        if self._type is not None:
            typeSplit = self._type.split("_")
            spectralClass = typeSplit[0]
        if self._subClass is not None:
            spectralClass = spectralClass + str(self._subClass)

        if self._luminosity is not None:
            spectralClass = spectralClass + ' ' + str(self._luminosity)

        return spectralClass

    # some simple getters
    def isMainStar(self):
        return self._isMainStar

    def getSubClass(self):
        return self._subClass

    def getAbsoluteMagnitude(self):
        return self._absoluteMagnitude

    def getAge(self):
        return self._age

    def getAgeAsString(self):
        if self._age is None:
            return ''

        age = self._age / 1000
        return locale.format_string('%.4f', age, True) + ' ' + QCoreApplication.translate("Systemview", "million_years")

    def getLuminosity(self):
        return self._luminosity

    def getRadiusAsString(self, decimals=4):
        if self._radius is None:
            return ""

        radius = Converter.getSunRadii(self._radius)
        numFormat = '%.' + str(decimals) + 'f'

        # return < 0.1 sun radius as km (useful for black holes, NS etc)
        if radius < 0.1:
            radius = round((self._radius / 1000), decimals)
            radius = locale.format_string(numFormat, radius)
            return str(radius) + " " + QCoreApplication.translate("Systemview", "kilometers_abbr")

        return locale.format_string(numFormat, radius, True) + ' ' + QCoreApplication.translate("Systemview", "sunradius_abbr")
