from typing import List
from PyQt5.QtCore import QCoreApplication
from decimal import *
import locale
from package.Journal.Events.BodyEntity import BodyEntity
from package.Journal.Events.Material import Material
from package.Journal.Events.AtmosphereElement import AtmosphereElement
from package.Helper.Converter import Converter
from package.Journal.Events.SSASignal import SSASignal
from package.Journal.Events.CodexEntry import CodexEntry
from package.Journal.Events.ScanOrganic import ScanOrganic


class Planet(BodyEntity):
    def __init__(self):
        super().__init__()

        self._isPlanet = True
        self._tidalLock = False
        self._terraformable = 0
        self._mass = None
        self._gravity = None
        self._surfacePressure = None
        self._atmosphere = None
        self._atmosphereType = None
        self._atmosphereComposition = []
        self._volcanism = None
        self._isLandable = False
        self._wasMapped = True
        self._detailScan = False
        self._efficentScan = False
        self._material = []
        self.signals = {
            SSASignal.DB_TYPE_BIO: 0,
            SSASignal.DB_TYPE_GEO: 0,
        }
        self.codex = {
            SSASignal.DB_TYPE_BIO: [],
            SSASignal.DB_TYPE_GEO: [],
            SSASignal.DB_TYPE_MISC: [],
        }
        self.organicScans = {
            ScanOrganic.SCAN_TYPE_LOG: [],
            ScanOrganic.SCAN_TYPE_ANALYSE: []
        }

    def initFromJournal(self, data):
        super().initFromJournal(data)

        self._mass = Decimal(self.getParam(data, 'MassEM'))
        self._gravity = Decimal(self.getParam(data, 'SurfaceGravity'))
        self._surfacePressure = self.getDecimal(self.getParam(data, 'SurfacePressure'))
        self._atmosphere = self.getParam(data, 'Atmosphere')
        self._atmosphereType = self.getParam(data, 'AtmosphereType')
        self._volcanism = self.getParam(data, 'Volcanism')
        self._type = self.getParam(data, 'PlanetClass', '')

        # TerraformState has different options, better save them all
        if self.getParam(data, 'TerraformState') == "Terraformable":
            self._terraformable = 1
        elif self.getParam(data, 'TerraformState') == "Terraforming":
            self._terraformable = 2
        elif self.getParam(data, 'TerraformState') == "Terraformed":
            self._terraformable = 3

        self._wasDiscovered = True
        if (self.getParam(data, 'WasDiscovered') is False) and (self.getParam(data, 'WasMapped') is False):
            self._wasDiscovered = False

        self._wasMapped = False
        if self.getParam(data, 'WasMapped') == True:
            self._wasMapped = True

        self._isLandable = False
        if self.getParam(data, 'Landable') == True:
            self._isLandable = True

        self._tidalLock = False
        if self.getParam(data, 'TidalLock') == True:
            self._tidalLock = True

        self.initMaterialsFromJournal(data)
        self.initAtmosphereCompositionFromJournal(data)

    def initFromDatabase(self, data):
        super().initFromDatabase(data)

        self._mass = Decimal(data['mass'])
        self._gravity = Decimal(data['gravity'])
        self._terraformable = data['terraformable']
        if data['surface_pressure'] is not None:
            self._surfacePressure = Decimal(data['surface_pressure'])
        self._atmosphere = data['atmosphere']
        self._atmosphereType = data['atmosphere_type']
        self._volcanism = data['volcanism']

        self._tidalLock = False
        if data['tidal_lock'] == 1:
            self._tidalLock = True

        self._isLandable = False
        if data['landable'] == 1:
            self._isLandable = True

        self._wasMapped = False
        if data['was_mapped'] == 1:
            self._wasMapped = True

        self._detailScan = False
        if data['detail_scan'] == 1:
            self._detailScan = True

        self._efficentScan = False
        if data['efficient_scan'] == 1:
            self._efficentScan = True

    def initMaterialsFromJournal(self, data):
        materials = self.getParam(data, "Materials", [])

        for materialData in materials:
            mat = Material()
            mat.initFromJournal(materialData)
            self._material.append(mat)

    def initAtmosphereCompositionFromJournal(self, data):
        elements = self.getParam(data, "AtmosphereComposition", [])

        for elementData in elements:
            element = AtmosphereElement()
            element.initFromJournal(elementData)
            self._atmosphereComposition.append(element)

    def initSignalsFromDb(self, signals):
        for signal in signals:
            if signal['type'] == SSASignal.JOURNAL_TYPE_BIO:
                self.signals[SSASignal.DB_TYPE_BIO] = signal['amount']
            elif signal['type'] == SSASignal.JOURNAL_TYPE_GEO:
                self.signals[SSASignal.DB_TYPE_GEO] = signal['amount']

    def addSignalsFromJournal(self, signals):
        for signal in signals:
            self.signals[signal.getDbType()] = signal.getCount()

    def initMaterialsFromDatabase(self, materials: list) -> None:
        for materialData in materials:
            mat = Material()
            mat.initFromDatabase(materialData)
            self._material.append(mat)

    def initAtmosphereCompositionFromDatabase(self, data) -> None:
        for entry in data:
            element = AtmosphereElement()
            element.initFromDatabase(entry)
            self._atmosphereComposition.append(element)

    def isTerraformable(self):
        return self._terraformable > 0

    def addCodexEntry(self, codex: CodexEntry) -> None:
        if codex.getSubCategory() == "$Codex_SubCategory_Organic_Structures;":
            self.codex[SSASignal.DB_TYPE_BIO].append(codex)
        elif codex.getSubCategory() == "$Codex_SubCategory_Geology_and_Anomalies;":
            self.codex[SSASignal.DB_TYPE_GEO].append(codex)
        else:
            self.codex[SSASignal.DB_TYPE_MISC].append(codex)

    def getCodexEntries(self, codexType: str) -> List[CodexEntry]:
        return self.codex[codexType]

    def addOrganicScan(self, scan: ScanOrganic) -> None:
        if scan.getScanType() == ScanOrganic.SCAN_TYPE_LOG:
            self.organicScans[ScanOrganic.SCAN_TYPE_LOG].append(scan)
        elif scan.getScanType() == ScanOrganic.SCAN_TYPE_ANALYSE:
            self.organicScans[ScanOrganic.SCAN_TYPE_ANALYSE].append(scan)

    def getOrganicScans(self, scanType: str) -> List[ScanOrganic]:
        return self.organicScans[scanType]

    # some simple getters
    def isTidalLocked(self):
        return self._tidalLock

    def getGravity(self):
        return self._gravity

    def getGravityAsString(self, decimals=4):
        numFormat = '%.' + str(decimals) + 'f'
        grav = Converter.getGravity(self._gravity)

        return locale.format_string(numFormat, grav, True) + " " + QCoreApplication.translate("Systemview", "gravity_abbr")

    def getSurfacePressure(self):
        return self._surfacePressure

    def getAtmosphere(self):
        return self._atmosphere

    def getAtmosphereType(self):
        return self._atmosphereType

    def getVolcanism(self):
        return self._volcanism

    def isLandable(self):
        return self._isLandable

    def wasMapped(self):
        return self._wasMapped

    def isDetailScaned(self):
        return self._detailScan

    def isEfficientScan(self):
        return self._efficentScan

    def getRadiusAsString(self, decimals=4):
        if self._radius is None:
            return ""

        numFormat = '%.' + str(decimals) + 'f'
        rad = self._radius / 1000
        return locale.format_string(numFormat, rad, True) + " " + QCoreApplication.translate("Systemview", "kilometers_abbr")

    def getMaterial(self) -> List[Material]:
        return self._material

    def getTerraformState(self):
        return self._terraformable

    def getAtmosphereComposition(self) -> List[AtmosphereElement]:
        return self._atmosphereComposition

    def getBodySignalKey(self):
        return str(self._systemAddress) + '_' + str(self._bodyId)

    def getBioSignalCount(self):
        return self.signals[SSASignal.DB_TYPE_BIO]

    def getGeoSignalCount(self):
        return self.signals[SSASignal.DB_TYPE_GEO]
