import json
import pytz
import locale
from typing import List
from PyQt5.QtCore import QCoreApplication
from decimal import *
from package.Journal.Events.BaseEntity import BaseEntity
from package.Journal.Events.Ring import Ring
from package.Helper.Converter import Converter
from package.Elite.Enum import LIGHT_SECOND_M


class BodyEntity(BaseEntity):
    def __init__(self):
        super().__init__()
        self._systemAddress = None
        self._systemName = None
        self._bodyId = None
        self._name = None
        self._type = None
        self._distance = None
        self._radius = None
        self._temperature = None
        self._semiMajorAxis = None
        self._eccentricity = None
        self._inclination = None
        self._periapsis = None
        self._orbitalPeriod = None
        self._rotationPeriod = None
        self._axialTilt = None
        self._wasDiscovered = True
        self._discoveryDate = None
        self._isStar = False
        self._isPlanet = False
        self._internalId = None
        self._parents = []
        self._rings = []
        self._temperatureFormat = 'K'

    def initFromJournal(self, data):
        super().initFromJournal(data)
        self._systemAddress = self.getParam(data, 'SystemAddress', '')
        self._systemName = self.getParam(data, 'StarSystem', '')
        self._bodyId = self.getParam(data, 'BodyID', 0)
        self._name = self.getParam(data, 'BodyName', '')
        self._type = self.getParam(data, 'StarType')
        self._distance = Decimal(self.getParam(data, 'DistanceFromArrivalLS', -1))
        self._radius = Decimal(self.getParam(data, 'Radius', 0))
        self._temperature = Decimal(self.getParam(data, 'SurfaceTemperature', 0))
        self._semiMajorAxis = Decimal(self.getParam(data, 'SemiMajorAxis', 0))
        self._eccentricity = Decimal(self.getParam(data, 'Eccentricity', 0))
        self._inclination = Decimal(self.getParam(data, 'OrbitalInclination', 0))
        self._periapsis = Decimal(self.getParam(data, 'Periapsis', 0))
        self._orbitalPeriod = Decimal(self.getParam(data, 'OrbitalPeriod', 0))
        self._rotationPeriod = Decimal(self.getParam(data, 'RotationPeriod', 0))
        self._axialTilt = Decimal(self.getParam(data, 'AxialTilt', 0))
        self._wasDiscovered = self.getParam(data, 'WasDiscovered', True)
        self._parents = self.getParam(data, 'Parents', [])
        self.initRingsFromJournal(data)

    def initFromDatabase(self, data):
        # decimal can be initialized with Decimal(value or 0) if value = None
        if data is not None:
            self._systemAddress = data['system_address']
            self._bodyId = data['body_id']
            self._name = data['name']
            self._type = data['type']
            self._distance = Decimal(data['distance'])
            self._radius = Decimal(data['radius'])
            self._temperature = Decimal(data['temperature'])
            self._semiMajorAxis = Decimal(data['semi_major_axis'])
            self._eccentricity = Decimal(data['eccentricity'])
            self._inclination = Decimal(data['inclination'])
            self._periapsis = Decimal(data['periapsis'])
            self._orbitalPeriod = Decimal(data['orbital_period'])
            self._rotationPeriod = Decimal(data['rotation_period'])
            self._axialTilt = Decimal(data['axial_tilt'])
            self._discoveryDate = data['discovery_date']
            self._parents = json.loads(data['parents'])
            self._internalId = data['id']

            if 'system_name' in data:
                self._systemName = data['system_name']

            self._wasDiscovered = False
            if data['was_discovered'] == 1:
                self._wasDiscovered = True

    def initRingsFromJournal(self, data):
        rings = self.getParam(data, "Rings", [])

        for ringData in rings:
            # we want no asteroid belts for now
            if "Ring" not in ringData['Name']:
                continue

            ring = Ring()
            ring.initFromJournal(ringData)
            self._rings.append(ring)

    def initRingsFromDatabase(self, rings):
        for ringData in rings:

            ring = Ring()
            ring.initFromDatabase(ringData)
            self._rings.append(ring)

    def stripZeros(self, value):
        if "." in str(value):
            return str(value).rstrip('0').rstrip('.')
        else:
            return str(value)

    def getLargestRing(self) -> Ring:
        largest = None
        for ring in self.getRings():
            if largest is None:
                largest = ring

            if ring.getRingWidth() > largest.getRingWidth():
                largest = ring

        return largest

    # simple getters
    def isStar(self):
        return self._isStar

    def isPlanet(self):
        return self._isPlanet

    def getSystemAddress(self):
        return self._systemAddress

    def getSystemName(self):
        return self._systemName

    def getBodyId(self):
        return self._bodyId

    def getName(self):
        return self._name

    def getType(self):
        return self._type

    def getDistance(self):
        return self._distance

    def getDistanceAsString(self, decimals=4):
        if self._distance is None:
            return ''

        numFormat = '%.' + str(decimals) + 'f'
        return locale.format_string(numFormat, self._distance, True) + " " + QCoreApplication.translate("Systemview", "lightsecond_abbr")

    def getMass(self):
        return self._mass

    def getMassAsString(self, decimals=4):
        if self._mass is None:
            return ''

        numFormat = '%.' + str(decimals) + 'f'
        return locale.format_string(numFormat, self._mass, True)

    def getRadius(self):
        return self._radius

    def getTemperature(self):
        return self._temperature

    def setTemperatureFormat(self, format) -> None:
        self._temperatureFormat = format

    def getTemperatureAsString(self, decimals: int=4) -> str:
        if self._temperatureFormat == 'K':
            return self.getTemperatureAsKelvin(decimals)
        else:
            return self.getTemperatureAsCelsius(decimals)

    def getTemperatureAsKelvin(self, decimals: int=4) -> str:
        if self._temperature is None:
            return ""

        numFormat = '%.' + str(decimals) + 'f'
        return locale.format_string(numFormat, self.getTemperature(), True) + " K"

    def getTemperatureAsCelsius(self, decimals=4):
        if self._temperature is None:
            return ""

        cel = Converter.celvin2Celsius(self._temperature)
        numFormat = '%.' + str(decimals) + 'f'
        return locale.format_string(numFormat, cel, True) + " C"

    def getSemiMajorAxis(self):
        return self._semiMajorAxis

    def getEccentricity(self):
        return self._eccentricity

    def getEccentricityAsString(self, decimals=4):
        if self._eccentricity is None:
            return ''

        numFormat = '%.' + str(decimals) + 'f'
        return locale.format_string(numFormat, self._eccentricity, True)

    def getInclination(self):
        return self._inclination

    def getInclinationAsString(self, decimals: int = 4) -> str:
        if self._inclination is None:
            return ""

        numFormat = '%.' + str(decimals) + 'f'
        return locale.format_string(numFormat, self._inclination, True) + " " + \
               QCoreApplication.translate("Systemview", "degree_symbol")

    def getPeriapsis(self):
        return self._periapsis

    def getOrbitalPeriod(self):
        return self._orbitalPeriod

    def getOrbitalPeriodAsString(self):
        if self._orbitalPeriod is None:
            return ""

        return Converter.getPeriod(self._orbitalPeriod)

    def getRotationPeriod(self):
        return self._rotationPeriod

    def getRotationPeriodAsString(self):
        if self._rotationPeriod is None:
            return ''
        return Converter.getPeriod(self._rotationPeriod)

    def getAxialTilt(self):
        return self._axialTilt

    def getAxialTiltAsString(self, decimals: int = 4) -> str:
        if self._axialTilt is None:
            return ""

        numFormat = '%.' + str(decimals) + 'f'
        return locale.format_string(numFormat, self._axialTilt, True) + " " + \
               QCoreApplication.translate("Systemview", "degree_symbol")

    def wasDiscovered(self):
        return self._wasDiscovered

    def hasRings(self):
        return len(self._rings) > 0

    def getRings(self) -> List[Ring]:
        return self._rings

    def getDiscoveryDate(self):
        return self._discoveryDate.replace(tzinfo=pytz.utc).astimezone()

    def getFormatedDiscoveryDate(self, format: str = "%d.%m.%Y %H:%M:%S"):
        return self.getDiscoveryDate().strftime(format)

    def getInternalId(self):
        return self._internalId

    def getParents(self):
        return self._parents

    def getParentStar(self):
        parentStar = None
        for pair in self._parents:
            if "Star" in pair:
                parentStar = pair['Star']

        return parentStar

    def getParentPlanet(self):
        parentPlanet = None
        for pair in self._parents:
            if 'Planet' in pair and parentPlanet is None:
                parentPlanet = pair['Planet']

        return parentPlanet

    def getNullParent(self):
        nullParent = None
        if len(self._parents) > 0 and "Null" in self._parents[0]:
            nullParent = self._parents[0]

        return nullParent

    def getFirstParent(self):
        firstParent = None
        if len(self._parents) > 0:
            firstParent = self._parents[0]

        return firstParent

    def getParentId(self):
        parentId = None
        if len(self._parents) > 0:
            if 'Planet' in self._parents[0]:
                parentId = self._parents[0]['Planet']

            if 'Star' in self._parents[0]:
                parentId = self._parents[0]['Star']

        return parentId

    def isBinaryObject(self):
        isBinary = False
        if (len(self._parents) > 0) and ('Null' in self._parents[0]):
            isBinary = True

        return isBinary

    def getSemiMajorAxisAsString(self, decimals=4):
        numFormat = '%.' + str(decimals) + 'f'
        distanceParent = self.getSemiMajorAxis() / Decimal(LIGHT_SECOND_M)
        return str(locale.format_string(numFormat, distanceParent, True)) + " " + QCoreApplication.translate("Systemview", "lightsecond_abbr")

    def isTerraformable(self) -> bool:
        """
        Dummy method to avoid confusing in ScanEvent if the object is a planet or star
        Method is overwritten in Planet class
        :return: bool
        """
        return False
