import logging
from abc import ABC, abstractmethod
from package.Database.DbFactory import DbFactory


logger = logging.getLogger(__name__)

class TableAbstract(ABC):
    def __init__(self):
        self.db = DbFactory.getDatabase()

        if DbFactory.isMySqlConnection():
            self.currentConnection = DbFactory.DB_MYSQL
        elif DbFactory.isSqliteConnection():
            self.currentConnection = DbFactory.DB_SQLITE
        else:
            raise EnvironmentError("Database type not found")

    def install(self):
        logger.info("Installing with type: %s", self.currentConnection)
        if self.currentConnection == DbFactory.DB_MYSQL:
            self.installMysql()
        elif self.currentConnection == DbFactory.DB_SQLITE:
            self.installSqlite()

    def updateTo(self, version):
        # first look if there is a general update method
        methodName = "updateTo" + str(version)

        if hasattr(self, methodName) and callable(getattr(self, methodName)):
            logger.info("calling general method: %s", methodName)
            getattr(self, methodName)()
        else:
            # no general method. call individual ones
            if self.currentConnection == DbFactory.DB_MYSQL:
                methodName += "Mysql"
            elif self.currentConnection == DbFactory.DB_SQLITE:
                methodName += "Sqlite"

            if hasattr(self, methodName) and callable(getattr(self, methodName)):
                logger.info("calling specific method: %s", methodName)

                getattr(self, methodName)()

    @abstractmethod
    def installMysql(self):
        pass

    @abstractmethod
    def installSqlite(self):
        pass
