import json
from package.Import.ImporterBody import ImporterBody
from package.Elite.Materials import MATERIALS
from package.Import.ImporterSAASignal import ImporterSAASignal


class ImporterPlanet(ImporterBody):
    def __init__(self):
        super().__init__()
        self.ssaImporter = ImporterSAASignal()
        self.bodySignalEvent = None

    def flush(self):
        super().flush()
        self.bodySignalEvent = None
        self.ssaImporter.flush()

    def setBodySignalEvent(self, event):
        self.bodySignalEvent = event

    def exists(self):
        bodyName = self.data.getName()
        exists = False

        if (self.systemAddress is not None) and (bodyName is not None):
            records = self.db.select("bodies", ["id"], {"system_address": self.systemAddress, "name": bodyName}, None, 1)

            if records == []:
                exists = False
            else:
                exists = True

        return exists

    def insert(self):
        wasDiscovered = 1
        wasMapped = 0
        landable = 0
        tidalLock = 0

        if (self.data.wasDiscovered() is False) and (self.data.wasMapped() is False):
            wasDiscovered = 0

        if self.data.wasMapped() == True:
            wasMapped = 1

        if self.data.isLandable() == True:
            landable = 1

        if self.data.isTidalLocked() == True:
            tidalLock = 1

        # star and/or planet parents
        parents = json.dumps(self.data.getParents())

        data = {
            "system_address": self.systemAddress,
            "name": self.data.getName(),
            "body_id": self.data.getBodyId(),
            "type": self.data.getType(),
            "distance": self.data.getDistance(),
            "mass": self.data.getMass(),
            "radius": self.data.getRadius(),
            "tidal_lock": tidalLock,
            "terraformable": self.data.getTerraformState(),
            "gravity": self.data.getGravity(),
            "temperature": self.data.getTemperature(),
            "surface_pressure": self.data.getSurfacePressure(),
            "atmosphere": self.data.getAtmosphere(),
            "atmosphere_type": self.data.getAtmosphereType(),
            "volcanism": self.data.getVolcanism(),
            "landable": landable,
            "semi_major_axis": self.data.getSemiMajorAxis(),
            "eccentricity": self.data.getEccentricity(),
            "inclination": self.data.getInclination(),
            "periapsis": self.data.getPeriapsis(),
            "orbital_period": self.data.getOrbitalPeriod(),
            "rotation_period": self.data.getRotationPeriod(),
            "axial_tilt": self.data.getAxialTilt(),
            "was_discovered": wasDiscovered,
            "was_mapped": wasMapped,
            "parents": parents,
            "discovery_date": self.data.getTimestamp()
        }

        bodyId = self.db.insert('bodies', data)
        self.insertMaterials(bodyId)
        self.insertRings(bodyId)
        self.insertAtmosphereComposition(bodyId)

        if self.bodySignalEvent is not None:
            self.insertBodySignals(bodyId)

    def insertMaterials(self, bodyId):
        allData = []
        materials = self.data.getMaterial()

        if len(materials) == 0:
            return

        for mat in materials:
            element = mat.getName()
            percent = mat.getPercent()
            materialId = MATERIALS[element]

            data = {
                "material_id": materialId,
                "body_id": bodyId,
                "percent": percent
            }
            allData.append(data)

        self.db.insertMany('material_bodies', allData)


    def insertRings(self, bodyId):
        if not self.data.hasRings():
            return

        systemAddress = self.systemAddress

        for ring in self.data.getRings():
            name = ring.getName()
            ringClass = ring.getType()
            mass = ring.getMass()
            innerRad = ring.getInnerRadius()
            outerRad = ring.getOuterRadius()

            # we want no asteroid belts for now
            if "Ring" not in name:
                continue

            data = {
                "system_address": systemAddress,
                "name": name,
                "ring_class": ringClass,
                "mass": mass,
                "inner_rad": innerRad,
                "outer_rad": outerRad
            }

            ringId = self.db.insert("rings", data)
            self.createBodyRingXref(bodyId, ringId)

    def createBodyRingXref(self, bodyId, ringId):
        data = {
            "ring_id": ringId,
            "body_id": bodyId
        }

        self.db.insert("body_rings", data)

    def setScannedStatus(self, entity):
        efficiency = 0

        probesUsed = entity.getProbesUsed()
        efficiencyTarget = entity.getEfficiencyTarget()
        if probesUsed <= efficiencyTarget:
            efficiency = 1

        updateData = {
            "detail_scan": 1,
            "efficient_scan": efficiency
        }

        where = {
            "name": entity.getBodyName(),
            "system_address": entity.getSystemAddress()
        }

        self.db.update("bodies", updateData, where)

    def insertAtmosphereComposition(self, bodyId):
        if self.data.isLandable():
            allData = []
            atmosphereComposition = self.data.getAtmosphereComposition()

            if len(atmosphereComposition) == 0:
                return

            for element in atmosphereComposition:
                data = {
                    "body_id": bodyId,
                    "element": element.getName(),
                    "percent": element.getPercent()
                }

                allData.append(data)

            self.db.insertMany('bodies_atmosphere', allData)

    def insertBodySignals(self, bodyId):
        signals = self.bodySignalEvent.getSignals()

        self.ssaImporter.insertPlanetSignals(bodyId, signals)
