import os, json, sys, traceback
import importlib
from math import floor
from PyQt5 import QtCore
from package.Config.ConfigManager import ConfigManager
from package.Base.LineHandler import LineHandler
from package.Elite.Events import EVENTS
from package.Helper.FileHelper import FileHelper


class ImportHandler(QtCore.QObject):
    running = True
    updateTextSignal = QtCore.pyqtSignal(str)
    updateBarFileSignal = QtCore.pyqtSignal(int)
    updateBarAllSignal = QtCore.pyqtSignal(int)
    errorSignal = QtCore.pyqtSignal(str)
    finishedSignal = QtCore.pyqtSignal()

    def __init__(self):
        super().__init__()
        self.journalDir = ConfigManager.getConfigValue("journal_directory")
        self.lineHandler = None
        self.setupLineHandler()

    def stop(self):
        self.running = False

    def run(self):
        fileHelper = FileHelper()
        files = fileHelper.getSortedLogfiles(self.journalDir)
        filesCount = len(files)

        if filesCount == 0:
            self.errorSignal.emit("No Journal files found")
            return

        onePercent = 100 / filesCount

        c = 1
        for entry in files:
            if not self.running:
                break

            filePath = self.journalDir + entry

            self.updateTextSignal.emit("<b>trying to import:</b>" + entry)

            fileSize = os.path.getsize(filePath)
            if fileSize == 0:
                continue

            fileOnePercent = 100 / fileSize

            fo = open(filePath, 'r', encoding="utf8")
            byteCount = 0
            for line in fo:
                if len(line) == 0:
                    continue

                if not self.running:
                    break

                byteCount += len(line.encode('utf-8'))

                try:
                    logEntry = json.loads(line)
                    if not self.checkGameVersion(logEntry):
                        self.updateTextSignal.emit("Skipping old journal file")
                        break

                    self.lineHandler.processLine(logEntry)
                except:
                    msg = "Error: " + str(sys.exc_info()[0])
                    msg += traceback.format_exc()
                    msg += line
                    self.updateTextSignal.emit(msg)

                lineProgess = (byteCount * fileOnePercent)
                self.updateBarFileSignal.emit(floor(lineProgess))

            fo.close()
            self.updateBarFileSignal.emit(100)

            progress = (c * onePercent)
            self.updateBarAllSignal.emit(floor(progress))
            c += 1

        if self.running:
            self.updateBarAllSignal.emit(100)

        self.updateTextSignal.emit("Finished running")
        self.finishedSignal.emit()

    def setupLineHandler(self):
        packageNameBase = "package.Journal.Handler.Import."
        self.lineHandler = LineHandler()

        for eventName in EVENTS:
            moduleName = eventName + "Event"
            packageName = packageNameBase + moduleName

            spec = importlib.util.find_spec(packageName)
            if spec is not None:
                eventModule = importlib.import_module(packageName)
                classname = getattr(eventModule, moduleName)
                eventInstance = classname()

                self.lineHandler.registerSubscriber(eventInstance)

    def checkGameVersion(self, line):
        journalFileValid = True

        # journal files before version 2.3 can't be imported due to missing system addresses
        if "event" in line and line["event"] == "Fileheader":
            if line["gameversion"] < "2.3":
                journalFileValid = False

        return journalFileValid
