from package.Journal.Events.BodyEntity import BodyEntity
from package.Calculator.Calculator import Calculator
from package.Elite.PlanetTypes import PlanetTypes


class System:
    def __init__(self):
        self.name = ''
        self.posX = 0
        self.posY = 0
        self.posZ = 0
        self.bodies = {}
        self.planetCount = 0
        self.starCount = 0
        self.scanComplete = False
        self.playerPosition = []
        self.distance = 0
        self.systemValue = 0
        self.terraformables = 0
        self.waterworlds = 0
        self.earthlikes = 0
        self.ammoniaworlds = 0
        self.detailScans = 0
        self.systemAddress = None
        self.stellarPhenomena = []

    def getSystemTree(self) -> dict:
        """
        returns all system bodies in a tree format, where the childs of each objects are available
        :return: dict
        """
        sortedBodies = sorted(self.bodies.items(), key=lambda x: x[0])
        sortedBodies = dict(sortedBodies)

        system = {}
        for bodyId in sortedBodies:
            bodyId = sortedBodies[bodyId].getBodyId()

            starParent = sortedBodies[bodyId].getParentStar()

            tmp = {
                "object": sortedBodies[bodyId],
                "childs": {}
            }

            # check if star is actually a child of another star
            if starParent is not None and starParent in system:
                system[starParent]["childs"][bodyId] = tmp
            else:
                system[bodyId] = tmp

        return system

    def processSystem(self) -> None:
        """
        Loops through each object and collects data
        :return:
        """
        for bodyId in self.bodies:
            object = self.bodies[bodyId]

            if object.isStar():
                bodyValue = Calculator.calculate(object)
            else:
                bodyValue = Calculator.calculate(object, object.isDetailScaned(), object.isEfficientScan())

            self.systemValue += bodyValue

            if object.getType() == PlanetTypes.TYPES['TYPE_WATER_WORLD']:
                self.waterworlds += 1

            if object.getType() == PlanetTypes.TYPES['TYPE_AMMONIA_WORLD']:
                self.ammoniaworlds += 1

            if object.getType() == PlanetTypes.TYPES['TYPE_EARTHLIKE']:
                self.earthlikes += 1

            if object.isPlanet() and object.isTerraformable():
                self.terraformables += 1

            if object.isPlanet() and object.isDetailScaned():
                self.detailScans += 1

    def hasNotableObjects(self) -> bool:
        """
        Returns if the system has any notable object
        (that is: WW, ELW, AW, terraformer)
        :return: bool
        """
        if self.getWaterWorldCount() == 0 \
                and self.getTerraformableCount() == 0 \
                and self.getAmmoniaWorldCount() == 0 \
                and self.getEartlikesCount() == 0:
            return False

        return True

    #
    # basic getter/setter
    #
    def setName(self, name: str) -> None:
        self.name = name

    def getName(self) -> str:
        return self.name

    def setPosX(self, pos: float) -> None:
        self.posX = pos

    def setPosY(self, pos: float) -> None:
        self.posY = pos

    def setPosZ(self, pos: float) -> None:
        self.posZ = pos

    def getPosX(self) -> float:
        return self.posX

    def getPosY(self) -> float:
        return self.posY

    def getPosZ(self) -> float:
        return self.posZ

    def addBody(self, bodyId: int, body: BodyEntity) -> None:
        self.bodies[bodyId] = body

        if body.isStar():
            self.starCount += 1

        if body.isPlanet():
            self.planetCount += 1

    def getBodies(self) -> dict:
        return self.bodies

    def getBodyById(self, bodyId: int) -> BodyEntity:
        body = None
        if bodyId in self.bodies:
            body = self.bodies[bodyId]

        return body

    def setPlanetCount(self, count: int) -> None:
        self.planetCount = count

    def getPlanetCount(self) -> int:
        return self.planetCount

    def setStarCount(self, count: int) -> None:
        self.starCount = count

    def getStarCount(self) -> int:
        return self.starCount

    def getBodiesCount(self) -> int:
        return len(self.bodies)

    def setScanComplete(self, complete: bool) -> None:
        self.scanComplete = complete

    def isScanComplete(self) -> bool:
        return self.scanComplete

    def setPlayerPosition(self, pos: list) -> None:
        self.playerPosition = pos

    def getPlayerPosition(self) -> list:
        return self.playerPosition

    def setDistance(self, distance: float) -> None:
        self.distance = distance

    def getEstimatedValue(self) -> int:
        return self.systemValue

    def getDistance(self) -> float:
        return self.distance

    def getWaterWorldCount(self) -> int:
        return self.waterworlds

    def getAmmoniaWorldCount(self) -> int:
        return self.ammoniaworlds

    def getEartlikesCount(self) -> int:
        return self.earthlikes

    def getTerraformableCount(self) -> int:
        return self.terraformables

    def getDetailScans(self) -> int:
        return self.detailScans

    def getSystemAddress(self) -> int:
        return self.systemAddress

    def setSystemAddress(self, systemAddress: int) -> None:
        self.systemAddress = systemAddress

    def getStellarPhenomena(self) -> list:
        return self.stellarPhenomena

    def addStellarPhenomena(self, name: str) -> None:
        self.stellarPhenomena.append(name)
