from package.Database.DbFactory import DbFactory
from package.Journal.Events.ScanOrganic import ScanOrganic


class PlanetReader:
    def getPlanetsForSystem(self, systemAddress):
        db = DbFactory.getDatabase()
        query = "select b.*, s.name as system_name from bodies b" \
                " left join systems s on b.system_address = s.system_address "\
                " where b.system_address = ?" \
                " order by b.body_id ASC"

        result = db.selectPlain(query, [systemAddress])
        if len(result) > 0:
            return result
        else:
            return []

    def getBodyId(self, bodyName):
        db = DbFactory.getDatabase()
        result = db.select("bodies", ["id"], {"name": bodyName}, None, 1)
        if len(result) > 0:
            return result[0]["id"]
        else:
            return None

    def getPlanetsWorthScanning(self, systemAddress):
        db = DbFactory.getDatabase()
        query = "select b.*, s.name as system_name from bodies b" \
                " left join systems s on b.system_address = s.system_address "\
                " where b.system_address = ? and " \
                "  (b.terraformable = 1 or b.type in ('Water world', 'Earthlike body', 'Ammonia world'))"

        result = db.selectPlain(query, [systemAddress])
        if len(result) > 0:
            return result
        else:
            return []

    def getPlanetByName(self, name):
        db = DbFactory.getDatabase()
        result = db.select("bodies", ["*"], {'name': name})
        if len(result) > 0:
            return result
        else:
            return []

    def getDetailScanCount(self):
        db = DbFactory.getDatabase()
        result = db.select('bodies', ['count(id) as scans'], {"detail_scan": 1})

        return result[0]['scans']

    def getSignalsForPlanet(self, planetId):
        db = DbFactory.getDatabase()

        query = """select s.type, s.amount from ssa_signals_planet p
                    left join ssa_signals s on s.id = p.ssa_id
                    where p.planet_id = ?
                """
        return db.selectPlain(query, [planetId])

    def getRingsForPlanet(self, planetId):
        db = DbFactory.getDatabase()

        query = """select r.id, r.system_address, r.name, r.ring_class, r.mass, r.inner_rad, r.outer_rad from body_rings b
                    left join rings r on b.ring_id = r.id
                    where b.body_id = ?
                """
        return db.selectPlain(query, [planetId])

    def searchPlanetByName(self, name):
        db = DbFactory.getDatabase()
        query = "select b.body_id as body_id, b.name as planet_name, s.system_address as system_address, s.name as system_name " \
                "from bodies b " \
                "left join systems s on b.system_address = s.system_address " \
                "where b.name like ? limit 10"
        return db.selectPlain(query, [name + "%"])

    def getMaterialsForPlanet(self, planetId: int) -> list:
        db = DbFactory.getDatabase()

        query = """select m.id, m.name, m.name_de, mb.percent 
                    from material_bodies mb 
                    left join materials m on m.id = mb.material_id 
                    where mb.body_id = ?
                    order by mb.percent desc
                """
        return db.selectPlain(query, [planetId])

    def getSsaSignalsForRing(self, planetId: int) -> list:
        db = DbFactory.getDatabase()

        query = """select s.id, s.type, s.type_localised, s.amount, sr.ring_id
                    from ssa_signals s 
                    left join ssa_signals_ring sr on sr.ssa_id = s.id
                    left join body_rings br on br.ring_id = sr.ring_id
                    where br.body_id = ?
                    order by s.amount desc
                """
        return db.selectPlain(query, [planetId])

    def getAtmosphereForPlanet(self, planetId: int) -> list:
        db = DbFactory.getDatabase()
        query = "select element, percent from bodies_atmosphere where body_id = ? order by percent desc"

        return db.selectPlain(query, [planetId])

    def getCodexEntriesForBody(self, planetId: int) -> list:
        db = DbFactory.getDatabase()
        query = "select * from codex where body_id = ? order by id asc"

        return db.selectPlain(query, [planetId])

    def getOrganicScansForBody(self, planetId: int) -> list:
        db = DbFactory.getDatabase()
        query = """select * from organic_samples
                    where body_id = ? and
                      (scan_type = ? or scan_type = ?)  
                """

        return db.selectPlain(query, [planetId, ScanOrganic.SCAN_TYPE_LOG, ScanOrganic.SCAN_TYPE_ANALYSE])
